package com.atlassian.logging.log4j.layout;

import java.io.IOException;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;
import java.util.TreeMap;

import com.atlassian.json.marshal.Jsonable;
import com.atlassian.logging.log4j.StackTraceCompressor;
import com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider;
import com.atlassian.logging.log4j.layout.json.JsonContextData;
import com.atlassian.logging.log4j.layout.json.JsonDataProvider;
import com.atlassian.logging.log4j.layout.json.JsonStaticData;

import com.google.common.base.Joiner;

import org.apache.log4j.Layout;
import org.apache.log4j.helpers.LogLog;
import org.apache.log4j.spi.LocationInfo;
import org.apache.log4j.spi.LoggingEvent;
import org.codehaus.jackson.JsonFactory;
import org.codehaus.jackson.JsonGenerator;

/**
 * Produces json logs based on this specification:
 * https://pug.jira-dev.com/wiki/display/CP/RFC+-+Consistent+JSON+logging+format+for+application+logs
 * <p/>
 * This layout can eliminate unnecessary stacktrace lines
 * <p/>
 * Settings:
 * <p/>
 *  * dataProvider - path.to.DataProvider - defaults to JsonDataProvider.  Must extend JsonDataProviderInterface
 *      This class is used to inject data into the json structure.  Please see {@link com.atlassian.logging.log4j.layout.json.DefaultJsonDataProvider} for
 *      default implementation.
 *  <p/>
 *  * FilterFrames - List of class/package names to be filtered out from stacktrace.
 *  Please see {@link com.atlassian.logging.log4j.StackTraceCompressor.Builder#filteredFrames(String)} for the syntax.
 */
public class JsonLayout extends Layout
{
    private static final String TIME_ZONE = "UTC";
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss,S'Z'");

    private static final Joiner STACK_TRACE_JOINER = Joiner.on("\n");

    /**
     * Application should set this in the log4j properties.  Defaults to JsonDataProvider. This class is used to inject
     * data into the json structure.  Please see JsonDataProvider for default implementation.
     */
    private JsonDataProvider dataProvider;

    private final JsonFactory jsonFactory = new JsonFactory();

    private JsonStaticData staticData;

    private StackTraceCompressor stackTraceCompressor;

    /**
     * This is the master switch on whether filtering will be applied or not.  This also allows it to to switched on or
     * off at run time
     */
    private boolean filteringApplied = true;

    /**
     * How many lines to always show
     */
    private int minimumLines = 6;

    /**
     * Whether to show a summary of eluded lines
     */
    private boolean showEludedSummary = false;

    /**
     * Holds the list of filtered frames.
     */
    private String filteredFrames;

    private static final class JSON_KEYS
    {
        private static final String TIMESTAMP = "timestamp";
        private static final String LEVEL = "level";
        private static final String SERVICE_ID = "serviceId";
        private static final String PRODUCT = "product";
        private static final String HOST_NAME = "hostname";
        private static final String ENVIRONMENT = "env";
        private static final String PROCESS_ID = "pid";
        private static final String THREAD = "thread";
        private static final String LOGGER = "logger";
        private static final String MESSAGE = "message";

        private static final String ERROR = "err";
        private static final String ERROR_MESSAGE = "msg";
        private static final String STACK_TRACE = "stack";

        private static final String CONTEXT = "ctx";
        private static final String REQUEST_ID = "requestId";
        private static final String SESSION_ID = "sessionId";
        private static final String USER_KEY = "userKey";

        private static final String UNICORN = "unicorn";
        private static final String DATA_CENTER = "dc";
        private static final String RACK = "rack";

        private static final String LOCATION = "location";
        private static final String CLASS = "class";
        private static final String METHOD = "method";
        private static final String LINE_NUMBER = "line";

        private static final String EXTRA = "ext";
    }

    static
    {
        DATE_FORMAT.setTimeZone(TimeZone.getTimeZone(TIME_ZONE));
    }

    @Override
    public String format(final LoggingEvent event)
    {
        StringWriter stringWriter = new StringWriter();

        try
        {
            JsonGenerator g = jsonFactory.createJsonGenerator(stringWriter);
            g.writeStartObject();

            writeFields(event, g);

            g.writeEndObject();
            g.close();
        }
        catch (IOException e)
        {
            throw new RuntimeException("JsonLayout - Failed to format", e);
        }

        stringWriter.append("\n");
        return stringWriter.toString();
    }

    public void setDataProvider(final JsonDataProvider dataProvider)
    {
        this.dataProvider = dataProvider;
    }

    /**
     * Used for log4j.properties
     */
    public void setDataProvider(final String dataProviderClazz)
    {
        try
        {
            this.dataProvider = (JsonDataProvider) Class.forName(dataProviderClazz).newInstance();
        }
        catch (ClassNotFoundException e)
        {
            throw new RuntimeException("JsonDataProvider implementation not found", e);
        }
        catch (InstantiationException | IllegalAccessException e)
        {
            throw new RuntimeException("Failed to instantiate JsonDataProvider implementation", e);
        }
    }

    /**
     * Controls whether filtering will be applied to via this pattern
     *
     * @param filteringApplied the boolean flag
     */
    public void setFilteringApplied(boolean filteringApplied)
    {
        this.filteringApplied = filteringApplied;
    }

    /**
     * Sets how many first N lines of the stack trace are always shown
     *
     * @param minimumLines how many lines to always show
     */
    public void setMinimumLines(int minimumLines)
    {
        this.minimumLines = minimumLines;
    }

    /**
     * Allows the eluded line to be shown in horizontal summary form
     *
     * @param showEludedSummary true if they are to be shown
     */
    public void setShowEludedSummary(boolean showEludedSummary)
    {
        this.showEludedSummary = showEludedSummary;
    }

    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will not be written to the log.
     * <p/>
     * You can specify multiples frames separated by commas eg "org.apache.tomcat, org.hibernate"
     *
     * @param filteredFrames class name(s) or package name(s) to be filtered
     */
    public void setFilteredFrames(String filteredFrames)
    {
        this.filteredFrames = filteredFrames;
    }

    private void writeFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        writeBasicFields(event, g);

        writeThrowableFields(event, g);
        writeContextFields(event, g);
        writeLocationFields(event, g);
        writeUnicornFields(g);
        writeExtFields(event, g);
    }

    private void writeBasicFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        Date date = new Date(event.timeStamp);

        g.writeStringField(JSON_KEYS.TIMESTAMP, DATE_FORMAT.format(date));
        g.writeStringField(JSON_KEYS.LEVEL, event.getLevel().toString());
        g.writeStringField(JSON_KEYS.SERVICE_ID, staticData.getServiceId());
        g.writeStringField(JSON_KEYS.PRODUCT, staticData.getProductName());
        g.writeStringField(JSON_KEYS.HOST_NAME, dataProvider.getHostName());
        g.writeStringField(JSON_KEYS.ENVIRONMENT, staticData.getEnvironment());
        g.writeNumberField(JSON_KEYS.PROCESS_ID, staticData.getProcessId());
        g.writeStringField(JSON_KEYS.THREAD, event.getThreadName());
        g.writeStringField(JSON_KEYS.LOGGER, event.getLoggerName());
        g.writeStringField(JSON_KEYS.MESSAGE, event.getMessage().toString());
    }

    private void writeThrowableFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        String[] throwableStrRep = event.getThrowableStrRep();
        if (throwableStrRep == null || throwableStrRep.length == 0)
        {
            return;
        }

        g.writeObjectFieldStart(JSON_KEYS.ERROR);

        g.writeStringField(JSON_KEYS.ERROR_MESSAGE, event.getThrowableInformation().getThrowable().getMessage());

        if (filteringApplied)
        {
            StringBuffer stackTrace = new StringBuffer();
            stackTraceCompressor.filterStackTrace(stackTrace, throwableStrRep);
            g.writeStringField(JSON_KEYS.STACK_TRACE, stackTrace.toString());
        }
        else
        {
            g.writeStringField(JSON_KEYS.STACK_TRACE, STACK_TRACE_JOINER.join(throwableStrRep));
        }

        g.writeEndObject();
    }


    private void writeContextFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        final JsonContextData contextData = dataProvider.getContextData(event);
        if (!contextData.isEmpty())
        {
            g.writeObjectFieldStart(JSON_KEYS.CONTEXT);

            writeFieldIfSet(g, JSON_KEYS.REQUEST_ID, contextData.getRequestId());
            writeFieldIfSet(g, JSON_KEYS.SESSION_ID, contextData.getSessionId());
            writeFieldIfSet(g, JSON_KEYS.USER_KEY, contextData.getUserKey());

            g.writeEndObject();
        }
    }

    private void writeFieldIfSet(final JsonGenerator g, String key, String value) throws IOException
    {
        if (value != null)
        {
            g.writeStringField(key, value);
        }
    }

    private void writeLocationFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        LocationInfo location = event.getLocationInformation();

        g.writeObjectFieldStart(JSON_KEYS.LOCATION);

        writeFieldIfSet(g, JSON_KEYS.CLASS, location.getClassName());
        writeFieldIfSet(g, JSON_KEYS.METHOD, location.getMethodName());
        writeFieldIfSet(g, JSON_KEYS.LINE_NUMBER, location.getLineNumber());

        g.writeEndObject();
    }

    private void writeUnicornFields(final JsonGenerator g) throws IOException
    {
        if (staticData.getDataCenter() == null && staticData.getRack() == null)
        {
            return;
        }

        g.writeObjectFieldStart(JSON_KEYS.UNICORN);

        g.writeStringField(JSON_KEYS.DATA_CENTER, staticData.getDataCenter());
        g.writeStringField(JSON_KEYS.RACK, staticData.getRack());

        g.writeEndObject();
    }

    private void writeExtFields(final LoggingEvent event, final JsonGenerator g) throws IOException
    {
        final Map<String, String> dataTable = dataProvider.getExtraData(event);

        final Map<String, MapOrVal> map = new HashMap<>();

        for (Object entryObj : dataTable.entrySet())
        {
            final Map.Entry entry = (Map.Entry) entryObj;

            final String[] keySequence = ((String) entry.getKey()).split("\\.");
            final Object value = entry.getValue();

            Map<String, MapOrVal> currentMap = map;
            for (int i = 0; i < keySequence.length; i++)
            {
                final String key = keySequence[i];
                if (currentMap.containsKey(key))
                {
                    if (i == keySequence.length - 1) // If last key
                    {
                        LogLog.debug("Collision in MDC for key " + keySequence);
                    }
                    else
                    {
                        final MapOrVal mapOrVal = currentMap.get(key);
                        if (mapOrVal.isMap())
                        {
                            currentMap = mapOrVal.map;
                        }
                        else
                        {
                            LogLog.debug("Collision in MDC for key " + keySequence);
                            break;
                        }
                    }
                }
                else
                {
                    if (i == keySequence.length - 1) // If last key
                    {
                        currentMap.put(key, MapOrVal.newVal(value));
                        currentMap = map;
                    }
                    else
                    {
                        final MapOrVal newMapOrVal = MapOrVal.newMap();
                        currentMap.put(key, newMapOrVal);
                        currentMap = newMapOrVal.map;
                    }
                }
            }
        }

        if (map.size() == 0)
        {
            return;
        }

        g.writeObjectFieldStart(JSON_KEYS.EXTRA);
        writeExtFieldsFromMap(g, map);
        g.writeEndObject();
    }

    private void writeExtFieldsFromMap(final JsonGenerator g, final Map<String, MapOrVal> map) throws IOException
    {
        for (Map.Entry<String, MapOrVal> entry : map.entrySet())
        {
            if (entry.getValue().isMap())
            {
                g.writeObjectFieldStart(entry.getKey());
                writeExtFieldsFromMap(g, entry.getValue().map);
                g.writeEndObject();
            }
            else
            {
                Object object = entry.getValue().val;
                if (object instanceof Jsonable)
                {
                    Jsonable jsonobject = (Jsonable) object;
                    g.writeFieldName(entry.getKey());
                    StringWriter w = new StringWriter();
                    jsonobject.write(w);
                    g.writeRawValue(w.toString());
                }
                else
                {
                    g.writeObjectField(entry.getKey(), object);
                }
            }
        }
    }

    private static class MapOrVal
    {
        public final Object val;
        public final Map<String, MapOrVal> map;

        private MapOrVal(final Object val, final Map<String, MapOrVal> map)
        {
            this.val = val;
            this.map = map;
        }

        public static MapOrVal newMap()
        {
            return new MapOrVal(null, new TreeMap<String, MapOrVal>());
        }

        public static MapOrVal newVal(final Object val)
        {
            return new MapOrVal(val, null);
        }

        public boolean isMap()
        {
            return map != null;
        }
    }

    @Override
    public boolean ignoresThrowable()
    {
        return false;
    }

    @Override
    public void activateOptions()
    {
        if (dataProvider == null)
        {
            dataProvider = new DefaultJsonDataProvider(); // Set default implementation
        }

        stackTraceCompressor = StackTraceCompressor.defaultBuilder(minimumLines, showEludedSummary)
                .filteredFrames(filteredFrames)
                .build();

        staticData = dataProvider.getStaticData();
    }

}
