package com.atlassian.logging.log4j.appender.fluentd;

import java.io.IOException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.UnknownHostException;

import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.apache.log4j.helpers.LogLog;

public class FluentdHttpSender implements FluentdSender
{
    private static final int EXPECTED_RESPONSE_CODE = 200;
    /**
     * Limit the connection to fluentd. Each fluentd aggregator must serve plenty of log shippers
     * and because shippers use keep alive connections, we must limit to 1 per shipper
     * @see <a href="https://jira.atlassian.com/browse/CONFDEV-33985">CONFDEV-33985</a>
     */
    private static final int MAX_CONNECTIONS_PER_ROUTE = 1;

    private final String fluentdEndpoint;

    private final CloseableHttpClient httpClient = HttpClients
            .custom()
            .setMaxConnPerRoute(MAX_CONNECTIONS_PER_ROUTE)
            .build();

    public FluentdHttpSender(final String fluentdEndpoint)
    {
        // JUST a workaround, should remove this hostname->IP transformation once fluentd-collector is set up properly
        this.fluentdEndpoint = resolveHostName(fluentdEndpoint);
    }

    @Override
    public void send(final String payload) throws FluentdRetryableException
    {
        final HttpPost httpPost = new HttpPost(fluentdEndpoint);

        HttpEntity reqEntity = MultipartEntityBuilder.create()
                .addPart("json", new StringBody(payload, ContentType.APPLICATION_JSON))
                .build();

        httpPost.setEntity(reqEntity);

        try
        {
            CloseableHttpResponse response = httpClient.execute(httpPost);
            try
            {
                EntityUtils.consume(response.getEntity()); // So that the connection can be reused.

                if (response.getStatusLine().getStatusCode() != EXPECTED_RESPONSE_CODE)
                {
                    throw new FluentdRetryableException("Bad status code return from fluentD: " + response.getStatusLine());
                }
            }
            finally
            {
                response.close();
            }
        }
        catch (IOException e)
        {
            throw new FluentdRetryableException(e);
        }
    }

    /**
     * This is JUST a workaround while waiting for a more robust and reliable way of DNS resolving for fluentd-collector
     * Ideally, we must not permanently stick to the first IP address from name server this way
     * @param endpoint URL to post log messages to
     * @return another URL in which hostname is replaced by IP address
     */
    private String resolveHostName(final String endpoint)
    {
        try
        {
            final String hostName = new URL(endpoint).getHost();
            for (final InetAddress address : InetAddress.getAllByName(hostName))
            {
                return endpoint.replace(hostName, address.getHostAddress());
            }
        }
        catch (MalformedURLException | UnknownHostException exc)
        {
            // nothing serious, just log then ignore it
            LogLog.error("Cannot resolve hostname of the endpoint", exc);
        }

        // Return original endpoint as default
        return endpoint;
    }
}
