package com.atlassian.logging.log4j;

import org.apache.log4j.helpers.PatternParser;
import org.apache.log4j.spi.ThrowableInformation;

/**
 * PatternLayout that looks to suffix all new lines contained within log messages with an indent for the new line.
 * <b>Note</b>
 * Unlike {@link com.atlassian.logging.log4j.FilteredPatternLayout} filtering for stack traces is disabled by default
 *
 * @since v1.2
 */
@SuppressWarnings ("UnusedDeclaration")
public class NewLineIndentingFilteringPatternLayout extends FilteredPatternLayout
{
    private String lineIndent;

    public NewLineIndentingFilteringPatternLayout()
    {
        this(CommonConstants.DEFAULT_NEW_LINE_PREFIX);
    }

    public NewLineIndentingFilteringPatternLayout(final String lineIndent)
    {
        setFilteringApplied(false);
        setLineIndent(lineIndent);
    }

    @Override
    protected PatternParser createPatternParser(final String pattern)
    {
        collapsingParser = new NewLineLogMessagePatternParser(pattern, lineIndent, getCategoryNameCollapsePrecision());
        return collapsingParser;
    }

    /**
     * A replacement for {@link org.apache.log4j.spi.ThrowableInformation#getThrowableStrRep()} which will split on any
     * newline characters that may have been injected into the exception message. This also replaces the call to
     * {@link java.lang.Throwable#printStackTrace(java.io.PrintStream)} which has hard coded Strings
     * ("\tat", "Caused By:", "\t... ") that are printed along with the exception message. If text is injected into the
     * exception message then these can be easily faked.
     * @param throwableInformation the throwable that is attached to the logging event
     * @return array containing all the StackTrace lines of the exception message
     */
    protected String[] getThrowableStrRep(final ThrowableInformation throwableInformation)
    {
        return new StackTraceInfo(throwableInformation.getThrowable(), lineIndent, isStackTracePackagingExamined()).getThrowableStrRep();
    }

    public String getLineIndent()
    {
        return lineIndent;
    }

    public void setLineIndent(final String lineIndent)
    {
        this.lineIndent = lineIndent;
        //Forces the parser to be recreated with the new indent settings
        this.setConversionPattern(getConversionPattern());
    }
}
