package com.atlassian.logging.log4j;

import org.apache.log4j.helpers.FormattingInfo;
import org.apache.log4j.helpers.PatternConverter;
import org.apache.log4j.helpers.PatternParser;
import org.apache.log4j.spi.LoggingEvent;

/**
 * PatternParser that will suffix newline characters with a custom separator (default four spaces ). This is
 * to better highlight that log messages are being injected when logging user entered information.
 *
 * @since v1.2
 */
public class NewLineLogMessagePatternParser extends PatternParser
{
    // Line ending combos include UNIX and MAC = \n, WINDOWS = \r\n, MAC pre OSX = \r the next major OS will need to choose \n\r
    private String lineIndent;

    public NewLineLogMessagePatternParser(final String pattern,
            final String lineIndent)
    {
        super(pattern);
        this.lineIndent = lineIndent;
    }

    @Override
    protected void finalizeConverter(final char c)
    {
        switch(c)
        {
            // m is the character the lo4j 1.2 uses to represent the log message passed along with
            // with the logging event
            case 'm':
                addConverter(new LineEndingStrippingPatternConverter(formattingInfo, lineIndent));
                break;
            default:
                super.finalizeConverter(c);
                break;
        }
    }

    /**
     * Append an indent to \r, \n, \r\n combination whilst preserving the original character
     */
    protected static class LineEndingStrippingPatternConverter extends PatternConverter
    {
        private final String lineIndent;

        LineEndingStrippingPatternConverter(final FormattingInfo info, final String lineIndent)
        {
            super(info);
            this.lineIndent = lineIndent;
        }

        @Override
        public String convert(LoggingEvent event)
        {
            return LogMessageUtil.appendLineIndent(event.getRenderedMessage(), lineIndent);
        }
    }
}
