package com.atlassian.logging.log4j;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Level;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.spi.LoggingEvent;
import org.apache.log4j.spi.ThrowableInformation;


/*
 * openutils for Log4j (http://www.openmindlab.com/lab/products/openutilslog4j.html) Copyright(C) null-2011, Openmind
 * S.r.l. http://www.openmindonline.it
 * <p/>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p/>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p/>
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */

/**
 * This is based off the original code at
 * <p/>
 * https://openutils.svn.sourceforge.net/svnroot/openutils/trunk/openutils-log4j/src/main/java/it/openutils/log4j/FilteredPatternLayout.java
 * <p/>
 * And hence the copyright repetition. Its deviated a fair but but the idea was theirs.
 * <p/>
 * To use this code you need to have this class as the PatternLayout and then to name the classes you wanted filtered out
 * <p/>
 * <pre>
 *
 * log4j.appender.filelog.layout=com.atlassian.logging.log4j.FilteredPatternLayout
 * log4j.appender.filelog.layout.FilterFrames=org.apache.catalina, org.apache.coyote, org.apache.tomcat.util.net, sun.reflect
 *
 *
 * </pre>
 */
@SuppressWarnings("UnusedDeclaration")
public class FilteredPatternLayout extends PatternLayout
{

    /**
     * Line separator for stacktrace frames.
     */
    private static String NL;

    static
    {
        try
        {
            NL = System.getProperty("line.separator");
        } catch (SecurityException ignore)
        {
            NL = "\n";
        }
    }

    /**
     * This is the master switch on whether filtering will be applied or not.  This also allows it to to switched on or
     * off at run time
     */
    private boolean filteringApplied = true;

    /**
     * This controls whether Debug level messages are filtered or not.  By default they are NOT.
     */
    private boolean filteringAppliedToDebugLevel = false;

    /**
     * Holds the list of filtered frames.
     */
    private final Set<String> filteredFrames = new HashSet<String>();

    /**
     * Holds the list of everyThingAfter filtered frames.
     */
    private final Set<String> filterEveryThingAfterFrames = new HashSet<String>();

    /**
     * Holds the list of frames that will generate a marker
     */
    private final Set<String> markerAtFrames = new HashSet<String>();

    /**
     * The message to output if we filter everything after that point
     */
    private String filterEveryThingAfterMessage = "\t\t(The rest of the stack trace has been filtered ...)";

    /**
     * The replacement string if a filter is applied to a stack trace line.  Default is "... "
     */
    private String filterReplacementToken = "... ";

    /**
     * The message to mark specific lines at
     */
    private String markerAtMessage = NL;

    /**
     * How many lines to always show
     */
    private int minimumLines = 6;

    /**
     * Whether to show a summary of eluded lines
     */
    private boolean showEludedSummary = false;

    /**
     * We must have a public no args constructor so that log4j can instantiate this class
     */
    public FilteredPatternLayout()
    {
    }


    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will not be written to the log.
     * <p/>
     * You can specify multiples frames separated by commas eg "org.apache.tomcat, org.hibernate"
     *
     * @param filterSpec a class name or package name to be filtered
     */
    public void setFilteredFrames(String filterSpec)
    {
        parseFilterSpec(filterSpec, filteredFrames);
    }

    /**
     * A pass of the stack frame is done and frames after the matching one are ignored and will not be written to the
     * log
     *
     * @param filterSpec a class name or package name to be filtered
     */
    public void setFilterEveryThingAfterFrames(String filterSpec)
    {
        parseFilterSpec(filterSpec, filterEveryThingAfterFrames);
    }

    /**
     * Any stack frame starting with <code>"at "</code> + <code>filter</code> will result in a marker message being
     * placed in the log
     *
     * @param filterSpec a class name or package name to cause a marker
     */
    public void setMarkerAtFrames(String filterSpec)
    {
        parseFilterSpec(filterSpec, markerAtFrames);
    }

    /**
     * @return true if the stack traces of Debug level log events are filtered out
     */
    public boolean isFilteringAppliedToDebugLevel()
    {
        return filteringAppliedToDebugLevel;
    }


    /**
     * If set to true the stack traces of Debug level log events are filtered out.  By default there are NOT, the
     * rationale being that if its a DEBUG log event you want everything.
     *
     * @param filteringAppliedToDebugLevel a boolean flag
     */
    public void setFilteringAppliedToDebugLevel(boolean filteringAppliedToDebugLevel)
    {
        this.filteringAppliedToDebugLevel = filteringAppliedToDebugLevel;
    }

    /**
     * @return the token string that is inserted when a stack trace lines is filtered out
     */
    public String getFilterReplacementToken()
    {
        return filterReplacementToken;
    }

    /**
     * Sets the token string that is inserted when a stack trace lines is filtered out
     *
     * @param filterReplacementToken the token string to be inserted
     */
    public void setFilterReplacementToken(String filterReplacementToken)
    {
        this.filterReplacementToken = StringUtils.defaultString(filterReplacementToken);
    }

    /**
     * @return the message that is output when all stack trace lines after a given line are ignored
     */
    public String getFilterEveryThingAfterMessage()
    {
        return filterEveryThingAfterMessage;
    }

    /**
     * Sets the message that is output when all stack trace lines after a given line are ignored
     *
     * @param filterEveryThingAfterMessage the string to be inserted
     */
    public void setFilterEveryThingAfterMessage(String filterEveryThingAfterMessage)
    {
        this.filterEveryThingAfterMessage = StringUtils.defaultString(filterEveryThingAfterMessage);
    }

    /**
     * @return the message that is output if a marker stack trace line is matched
     */
    public String getMarkerAtMessage()
    {
        return filterEveryThingAfterMessage;
    }

    /**
     * Sets the message that is output if a marker stack trace line is matched
     *
     * @param markerAtMessage the string to be inserted
     */
    public void setMarkerAtMessage(String markerAtMessage)
    {
        this.markerAtMessage = StringUtils.defaultString(markerAtMessage);
    }

    /**
     * @return true if filtering will be applied to an event
     */
    public boolean isFilteringApplied()
    {
        return filteringApplied;
    }

    /**
     * Controls whether filtering will be applied to via this pattern
     *
     * @param filteringApplied the boolean flag
     */
    public void setFilteringApplied(boolean filteringApplied)
    {
        this.filteringApplied = filteringApplied;
    }

    /**
     * @return shows the first N lines of the stack trace
     */

    public int getMinimumLines()
    {
        return minimumLines;
    }

    /**
     * Sets how many first N lines of the stack trace are always shown
     *
     * @param minimumLines how many lines to always show
     */
    public void setMinimumLines(int minimumLines)
    {
        this.minimumLines = minimumLines;
    }


    /**
     * @return true if the eluded liens are shown in summary form
     */
    public boolean isShowEludedSummary()
    {
        return showEludedSummary;
    }

    /**
     * Allows the eluded line to be shown in horizontal summary form
     *
     * @param showEludedSummary true if they are to be shown
     */
    public void setShowEludedSummary(boolean showEludedSummary)
    {
        this.showEludedSummary = showEludedSummary;
    }

    /**
     * This is really key to how this works.  Appenders by default output the stack trace UNLESS the pattern indicates
     * that it handles the exception by returning false here.
     *
     * @see org.apache.log4j.Layout#ignoresThrowable()
     */
    @Override
    public boolean ignoresThrowable()
    {
        return false;
    }

    /**
     * @see org.apache.log4j.PatternLayout#format(org.apache.log4j.spi.LoggingEvent)
     */
    @Override
    public String format(LoggingEvent event)
    {
        ThrowableInformation throwableInformation = event.getThrowableInformation();
        String superFormatted = super.format(event);
        if (throwableInformation == null)
        {
            return superFormatted;
        }
        return formatStackTrace(event, throwableInformation, superFormatted);
    }

    private String formatStackTrace(LoggingEvent event, ThrowableInformation throwableInformation, String formattedByPattern)
    {
        StringBuffer buffer = new StringBuffer(formattedByPattern);
        String[] stackTraceLines = getThrowableStrRep(throwableInformation);

        // if they have turned us off or its a debug event then go au natural
        if (!filteringApplied || (Level.DEBUG.equals(event.getLevel()) && !filteringAppliedToDebugLevel))
        {
            outputPlainThrowable(buffer, stackTraceLines);
        }
        else
        {
            outputFilteredThrowable(buffer, stackTraceLines);
        }

        return buffer.toString();
    }

    /**
     * A hook point for subclasses to override how the lines for an exception get generated.
     * @param throwableInformation The {@link org.apache.log4j.spi.ThrowableInformation} that is associated with the
     * logging event
     * @return array containing each of the log lines that will be written.
     */
    protected String[] getThrowableStrRep(final ThrowableInformation throwableInformation)
    {
        return throwableInformation.getThrowableStrRep();
    }

    private void outputPlainThrowable(StringBuffer buffer, String[] stackTraceLines)
    {
        for (String stackTraceLine : stackTraceLines)
        {
            buffer.append(stackTraceLine).append(NL);
        }
    }

    private void outputFilteredThrowable(StringBuffer buffer, String[] stackTraceLines)
    {
        int lineCount = 0;
        int filteredCount = 0;
        boolean ignoreLinesUntilEnd = false;
        boolean markerDue = false;
        List<String> eludedLineSummary = new ArrayList<String>(stackTraceLines.length);

        // show the first N lines
        lineCount = outputMinimumLines(buffer, stackTraceLines, lineCount, minimumLines);
        // now filter the rest
        for (; lineCount < stackTraceLines.length; lineCount++)
        {
            String stackTraceLine = stackTraceLines[lineCount];

            boolean filteredLine = false;
            if (causedBy(stackTraceLine))
            {
                // after a cause by line we reset and show the next min lines and then start filtering all over again
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                lineCount = outputMinimumLines(buffer, stackTraceLines, lineCount, minimumLines);
                ignoreLinesUntilEnd = false;
                continue;
            }
            else if (lineMatchesPattern(stackTraceLine, filterEveryThingAfterFrames))
            {
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                buffer.append(filterEveryThingAfterMessage);
                filteredCount = 0;
                ignoreLinesUntilEnd = true;
            }
            else if (lineMatchesPattern(stackTraceLine, filteredFrames))
            {
                filteredCount++;
                filteredLine = true;
            }

            if (ignoreLinesUntilEnd)
            {
                filteredLine = true;
            }

            if (!filteredLine)
            {
                appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
                if (markerDue)
                {
                    buffer.append(markerAtMessage).append(NL);
                    markerDue = false;
                }
                buffer.append(stackTraceLine);
                if (lineMatchesPattern(stackTraceLine, markerAtFrames))
                {
                    markerDue = true;
                }
                filteredCount = 0;
            }
            else
            {
                if (showEludedSummary)
                {
                    String summary = makeEludedSummary(stackTraceLine);
                    if (StringUtils.isNotBlank(summary))
                    {
                        eludedLineSummary.add(summary);
                    }
                }
            }
        }
        appendSkipIndicators(buffer, filteredCount, eludedLineSummary);
    }


    private int outputMinimumLines(StringBuffer buffer, String[] stackTraceLines, int lineCount, final int minimumLines)
    {
        int minLines = Math.min(lineCount + minimumLines, stackTraceLines.length);
        for (; lineCount < minLines; lineCount++)
        {
            String stackTraceLine = stackTraceLines[lineCount];
            buffer.append(stackTraceLine);
            if (lineCount < minLines - 1)
            {
                buffer.append(NL);
            }
        }
        return lineCount;
    }

    protected boolean causedBy(String stackTraceLine)
    {
        return stackTraceLine.startsWith("Caused by:");
    }

    private void appendSkipIndicators(StringBuffer buffer, int filteredCount, List<String> eludedLineSummary)
    {
        if (filteredCount > 0)
        {
            buffer.append("  <+").append(filteredCount).append(">");
        }
        if (showEludedSummary)
        {
            for (String summary : eludedLineSummary)
            {
                buffer.append(" ").append(summary);
            }
            eludedLineSummary.clear();
        }
        // if it does already have a newline on the end put one there
        int lastChar = buffer.length() - 1;
        if (buffer.lastIndexOf(NL) != lastChar)
        {
            buffer.append(NL);
        }
    }

    /**
     * Check if the given string starts with any of the filtered patterns.
     *
     * @param string    checked String
     * @param filterSet the set to cechk against
     * @return <code>true</code> if the begininning of the string matches a filtered pattern, <code>false</code>
     *         otherwise
     */
    private boolean lineMatchesPattern(String string, final Set<String> filterSet)
    {
        if (!filterSet.isEmpty())
        {
            Iterator<String> iterator = filterSet.iterator();
            while (iterator.hasNext())
            {
                if (string.trim().startsWith(iterator.next()))
                {
                    return true;
                }
            }
        }
        return false;
    }

    private void parseFilterSpec(String filterSpec, final Set<String> filterSet)
    {
        if (StringUtils.isNotBlank(filterSpec))
        {
            String[] split = filterSpec.split(",");
            for (String filter : split)
            {
                String trimmed = StringUtils.trim(filter);
                if (StringUtils.isNotBlank(trimmed))
                {
                    filterSet.add("at " + trimmed);
                }
            }
        }
    }

    private static Pattern CALL_SITE = Pattern.compile("(\\(.*:[0-9]+\\))");

    private String makeEludedSummary(String stackTraceLine)
    {
        Matcher matcher = CALL_SITE.matcher(stackTraceLine);
        if (matcher.find() && matcher.groupCount() >= 1) {
            return matcher.group(1);
        }
        return null;
    }

}
