package com.atlassian.net;

import java.net.InetAddress;
import java.net.UnknownHostException;

/**
 * Various network-related utilities.
 */
public final class NetworkUtils
{
    private NetworkUtils() {}


    /**
     * <p>
     * Returns the fully-qualified hostname (FQHN) of the current machine on a best-effort basis.
     * Otherwise this method follows the same contract as {@link InetAddress#getHostName()}, with
     * the caveat that the value of {@link InetAddress#getCanonicalHostName()} will be returned
     * if the returned hostname of {@link InetAddress#getLocalHost()} is "localhost".
     * </p>
     * <p>
     * This method's main reason for existence is to shim the changed behaviour
     * of {@link InetAddress#getHostName()} when called on {@link InetAddress#getLocalHost()}
     * in Java8 compared to Java7, in which the former can return "localhost" instead of
     * the FQHN that was returned in Java7.
     * </p>
     */
    public static final String getLocalHostName() throws UnknownHostException
    {
        InetAddress host = InetAddress.getLocalHost();
        String hostname = host.getHostName();

        if (hostname != null && !hostname.equals("localhost"))
        {
            return hostname;
        }

        return host.getCanonicalHostName();
    }
}
