package com.atlassian.asap.nimbus.parser;

import java.text.ParseException;

import com.atlassian.asap.api.JwsHeader.Header;
import com.atlassian.asap.api.JwtClaims.Claim;
import com.atlassian.asap.core.exception.JwtParseException;
import com.atlassian.asap.core.exception.MissingRequiredClaimException;
import com.atlassian.asap.core.exception.MissingRequiredHeaderException;
import com.atlassian.asap.core.exception.UnsupportedAlgorithmException;
import com.atlassian.asap.core.parser.JwtParser;
import com.atlassian.asap.core.parser.VerifiableJwt;

import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.JWSObject;
import com.nimbusds.jwt.JWTClaimsSet;

import net.minidev.json.JSONObject;

/**
 * A parser of JWT implemented using the Nimbus library.
 */
public class NimbusJwtParser implements JwtParser
{
    @Override
    public VerifiableJwt parse(String serializedJwt) throws JwtParseException, UnsupportedAlgorithmException
    {
        final JWSObject jwsObject;
        final JWTClaimsSet claims;

        try
        {
            jwsObject = JWSObject.parse(serializedJwt);

            validateRequiredHeaders(jwsObject);

            JSONObject jsonPayload = jwsObject.getPayload().toJSONObject();
            claims = JWTClaimsSet.parse(jsonPayload);
        }
        catch (ParseException e)
        {
            throw new JwtParseException(e);
        }

        validateRequiredClaims(claims);
        return NimbusVerifiableJwt.buildVerifiableJwt(jwsObject, claims);
    }

    private void validateRequiredHeaders(JWSObject jwsObject) throws MissingRequiredHeaderException
    {
        if (jwsObject.getHeader().getAlgorithm() == null || jwsObject.getHeader().getAlgorithm() == Algorithm.NONE)
        {
            throw new MissingRequiredHeaderException(Header.ALGORITHM);
        }

        if (jwsObject.getHeader().getKeyID() == null)
        {
            throw new MissingRequiredHeaderException(Header.KEY_ID);
        }
    }

    private void validateRequiredClaims(JWTClaimsSet claims) throws MissingRequiredClaimException
    {
        checkClaimNotNull(claims.getAudience(), Claim.AUDIENCE);
        checkClaimNotNull(claims.getIssuer(), Claim.ISSUER);
        checkClaimNotNull(claims.getJWTID(), Claim.JWT_ID);
        checkClaimNotNull(claims.getIssueTime(), Claim.ISSUED_AT);
        checkClaimNotNull(claims.getExpirationTime(), Claim.EXPIRY);
    }

    private static void checkClaimNotNull(Object claimValue, Claim claim) throws MissingRequiredClaimException
    {
        if (claimValue == null)
        {
            throw new MissingRequiredClaimException(claim);
        }
    }
}
