package com.atlassian.asap.core.keys.publickey;

import java.security.PublicKey;
import java.util.Iterator;
import java.util.List;

import com.atlassian.asap.api.exception.CannotRetrieveKeyException;
import com.atlassian.asap.core.exception.PublicKeyNotFoundException;
import com.atlassian.asap.core.keys.KeyProvider;
import com.atlassian.asap.core.validator.ValidatedKeyId;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Looks up keys sequentially in a given list of key providers.</p>
 *
 * <p>Note that the current implementation doesn't perform any caching, in
 * particular it doesn't remember where the key was found last time. That may
 * reduce the effectiveness of the provider's http cache if it comes later
 * in the keyProviderChain.</p>
 */
public class ChainedKeyProvider implements KeyProvider<PublicKey>
{
    private static final Logger logger = LoggerFactory.getLogger(ChainedKeyProvider.class);


    private final List<KeyProvider<PublicKey>> keyProviderChain;

    @VisibleForTesting
    ChainedKeyProvider(List<KeyProvider<PublicKey>> keyProviderChain)
    {
        this.keyProviderChain = ImmutableList.copyOf(keyProviderChain);
    }

    /**
     * Create a chained key provider representing a given list of key repository providers.
     * If there is only one provider in the chain, then return the provider instead of wrapping it.
     *
     * @param keyProviderChain set of chained key repository providers in order
     * @return the chained key provider if there are more than one providers, or a key provider when it is
     *          the only provider in the chain
     */
    public static KeyProvider<PublicKey> createChainedKeyProvider(List<KeyProvider<PublicKey>> keyProviderChain)
    {
        // this is an optimization to avoid wrapping the keyprovider
        // when there is only one keyprovider
        if (keyProviderChain.size() == 1)
        {
            return keyProviderChain.get(0);
        }
        else
        {
            return new ChainedKeyProvider(keyProviderChain);
        }
    }

    @Override
    public PublicKey getKey(ValidatedKeyId validatedKeyId) throws CannotRetrieveKeyException
    {
        Iterator<KeyProvider<PublicKey>> keyProviderIterator = keyProviderChain.iterator();
        while (keyProviderIterator.hasNext())
        {
            try
            {
                return keyProviderIterator.next().getKey(validatedKeyId);
            }
            catch (PublicKeyNotFoundException ex)
            {
                if (keyProviderIterator.hasNext())
                {
                    logger.debug("Key not found in the provider, going to failover to next provider", ex);
                }
                else
                {
                    // there is at least one key not found, but we don't know about the previous providers.
                    // thus log it as debug as this situation can be caused by invalid input
                    logger.debug("Error retrieving key from all key providers: {}", keyProviderChain, ex);
                    throw new CannotRetrieveKeyException("Error communicating with all key providers", ex);
                }
            }
            catch (CannotRetrieveKeyException ex)
            {
                if (keyProviderIterator.hasNext())
                {
                    logger.debug("Error retrieving key from the provider, going to failover to next provider", ex);
                }
                else
                {
                    // there was at least one communication failure, so log it as error
                    logger.error("Error retrieving key from all key providers: {}", keyProviderChain, ex);
                    throw new CannotRetrieveKeyException("Error communicating with all key providers", ex);
                }
            }

        }
        throw new CannotRetrieveKeyException("There are no key providers available in the chain");
    }

    /**
     * @return the chain of key providers in the same order as they are consulted to resolve a key
     */
    public List<KeyProvider<PublicKey>> getKeyProviderChain()
    {
        return keyProviderChain;
    }
}
