/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.asap.core.client.http;

import com.atlassian.asap.api.Jwt;
import com.atlassian.asap.api.JwtBuilder;
import com.atlassian.asap.api.JwtClaims;
import com.atlassian.asap.api.client.http.AuthorizationHeaderGenerator;
import com.atlassian.asap.api.exception.CannotRetrieveKeyException;
import com.atlassian.asap.api.exception.InvalidTokenException;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

public class CachingAuthorizationHeaderGenerator
implements AuthorizationHeaderGenerator {
    public static final int DEFAULT_CACHE_SIZE = 50;
    public static final Duration DEFAULT_EXPIRY = Duration.ofMinutes(10L);
    public static final Duration DEFAULT_EXPIRY_BUFFER = Duration.ofMinutes(1L);
    private static final Set<String> REGISTERED_CLAIMS = Collections.unmodifiableSet(Arrays.stream(JwtClaims.RegisteredClaim.values()).map(JwtClaims.RegisteredClaim::key).collect(Collectors.toSet()));
    private final AuthorizationHeaderGenerator delegate;
    private final Clock clock;
    private final Duration expiry;
    private final Duration expiryBuffer;
    private final Cache<JwtTemplate, CachedToken> tokenCache;

    public CachingAuthorizationHeaderGenerator(AuthorizationHeaderGenerator delegate) {
        this(delegate, Clock.systemUTC(), 50, DEFAULT_EXPIRY, DEFAULT_EXPIRY_BUFFER);
    }

    public CachingAuthorizationHeaderGenerator(AuthorizationHeaderGenerator delegate, Clock clock, int maxCacheSize, Duration expiry, Duration expiryBuffer) {
        this.delegate = Objects.requireNonNull(delegate);
        this.clock = Objects.requireNonNull(clock);
        this.expiry = Objects.requireNonNull(expiry);
        this.expiryBuffer = Objects.requireNonNull(expiryBuffer);
        Preconditions.checkArgument((boolean)expiryBuffer.minus(expiry).isNegative(), (Object)"Expiry must be larger than the expiry buffer");
        this.tokenCache = CacheBuilder.newBuilder().maximumSize((long)maxCacheSize).build();
    }

    public String generateAuthorizationHeader(Jwt jwt) throws InvalidTokenException, CannotRetrieveKeyException {
        CachedToken token;
        if (!this.isSuitableForReuse(jwt)) {
            return this.delegate.generateAuthorizationHeader(jwt);
        }
        JwtTemplate template = JwtTemplate.fromJwt(jwt);
        TokenCreator creator = () -> {
            Instant now = this.clock.instant();
            String value = this.delegate.generateAuthorizationHeader(template.toJwt(now, this.expiry));
            return new CachedToken(now.plus(this.expiry).minus(this.expiryBuffer), value);
        };
        try {
            token = (CachedToken)this.tokenCache.get((Object)template, (Callable)creator);
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            Throwables.propagateIfInstanceOf((Throwable)e.getCause(), InvalidTokenException.class);
            Throwables.propagateIfInstanceOf((Throwable)e.getCause(), CannotRetrieveKeyException.class);
            Throwables.propagateIfPossible((Throwable)e.getCause());
            throw Throwables.propagate((Throwable)e);
        }
        if (this.isRequiringRefresh(token)) {
            token = creator.call();
            this.tokenCache.put((Object)template, (Object)token);
        }
        return token.value;
    }

    private boolean isSuitableForReuse(Jwt jwt) {
        return jwt.getHeader().getAlgorithm() == JwtBuilder.DEFAULT_ALGORITHM && JwtBuilder.DEFAULT_LIFETIME.equals(Duration.between(jwt.getClaims().getIssuedAt(), jwt.getClaims().getExpiry())) && REGISTERED_CLAIMS.containsAll(jwt.getClaims().getJson().keySet());
    }

    private boolean isRequiringRefresh(CachedToken token) {
        return token == null || this.clock.instant().isAfter(token.expiry);
    }

    private static class CachedToken {
        private final Instant expiry;
        private final String value;

        CachedToken(Instant expiry, String value) {
            this.expiry = expiry;
            this.value = value;
        }
    }

    private static class JwtTemplate {
        private final String keyId;
        private final String issuer;
        private final Optional<String> subject;
        private final Set<String> audience;

        JwtTemplate(String keyId, String issuer, Optional<String> subject, Set<String> audience) {
            this.keyId = keyId;
            this.issuer = issuer;
            this.subject = subject;
            this.audience = audience;
        }

        static JwtTemplate fromJwt(Jwt jwt) {
            return new JwtTemplate(jwt.getHeader().getKeyId(), jwt.getClaims().getIssuer(), jwt.getClaims().getSubject(), jwt.getClaims().getAudience());
        }

        Jwt toJwt(Instant issuedAt, Duration expiry) {
            return JwtBuilder.newJwt().keyId(this.keyId).issuer(this.issuer).subject(this.subject).audience(this.audience).notBefore(Optional.of(issuedAt)).issuedAt(issuedAt).expirationTime(issuedAt.plus(expiry)).build();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            JwtTemplate that = (JwtTemplate)o;
            return this.keyId.equals(that.keyId) && this.issuer.equals(that.issuer) && this.subject.equals(that.subject) && this.audience.equals(that.audience);
        }

        public int hashCode() {
            return Objects.hash(this.keyId, this.issuer, this.subject, this.audience);
        }
    }

    @FunctionalInterface
    static interface TokenCreator
    extends Callable<CachedToken> {
        @Override
        public CachedToken call() throws InvalidTokenException, CannotRetrieveKeyException;
    }
}

