package com.atlassian.applinks.ui;

import com.atlassian.applinks.analytics.EntityLinksAdminViewEvent;
import com.atlassian.applinks.api.ApplicationType;
import com.atlassian.applinks.api.ApplicationTypeVisitor;
import com.atlassian.applinks.api.application.bamboo.BambooApplicationType;
import com.atlassian.applinks.api.application.bitbucket.BitbucketApplicationType;
import com.atlassian.applinks.api.application.confluence.ConfluenceApplicationType;
import com.atlassian.applinks.api.application.crowd.CrowdApplicationType;
import com.atlassian.applinks.api.application.fecru.FishEyeCrucibleApplicationType;
import com.atlassian.applinks.api.application.generic.GenericApplicationType;
import com.atlassian.applinks.api.application.jira.JiraApplicationType;
import com.atlassian.applinks.api.application.refapp.RefAppApplicationType;
import com.atlassian.applinks.core.util.MessageFactory;
import com.atlassian.applinks.host.spi.InternalHostApplication;
import com.atlassian.applinks.internal.common.docs.DocumentationLinker;
import com.atlassian.applinks.ui.auth.AdminUIAuthenticator;
import com.atlassian.applinks.ui.velocity.ListEntityLinksContext;
import com.atlassian.applinks.ui.velocity.VelocityContextFactory;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.webresource.WebResourceManager;
import com.atlassian.sal.api.auth.LoginUriProvider;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.websudo.WebSudoManager;
import com.atlassian.sal.api.websudo.WebSudoSessionException;
import com.atlassian.sal.api.xsrf.XsrfTokenAccessor;
import com.atlassian.sal.api.xsrf.XsrfTokenValidator;
import com.atlassian.templaterenderer.TemplateRenderer;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.Map;

import static java.util.Collections.singletonList;

/**
 * A servlet that renders a velocity template to display all configured entity links.
 *
 * @since 3.0
 */
public class ListEntityLinksServlet extends AbstractAppLinksAdminOnlyServlet {
    private static final String TEMPLATE_PATH = "com/atlassian/applinks/ui/admin/list_entity_links.vm";
    private final VelocityContextFactory velocityContextFactory;
    private final WebSudoManager webSudoManager;
    private final EventPublisher eventPublisher;

    public ListEntityLinksServlet(final I18nResolver i18nResolver,
                                  final MessageFactory messageFactory,
                                  final TemplateRenderer templateRenderer,
                                  final WebResourceManager webResourceManager,
                                  final InternalHostApplication internalHostApplication,
                                  final DocumentationLinker documentationLinker,
                                  final LoginUriProvider loginUriProvider,
                                  final AdminUIAuthenticator adminUIAuthenticator,
                                  final VelocityContextFactory velocityContextFactory,
                                  final WebSudoManager webSudoManager,
                                  final XsrfTokenAccessor xsrfTokenAccessor,
                                  final XsrfTokenValidator xsrfTokenValidator,
                                  final EventPublisher eventPublisher) {
        super(i18nResolver, messageFactory, templateRenderer,
                webResourceManager, adminUIAuthenticator, documentationLinker,
                loginUriProvider, internalHostApplication, xsrfTokenAccessor, xsrfTokenValidator);
        this.velocityContextFactory = velocityContextFactory;
        this.webSudoManager = webSudoManager;
        this.eventPublisher = eventPublisher;
    }

    @Override
    protected List<String> getRequiredWebResources() {
        return singletonList(WEB_RESOURCE_KEY + "list-entity-links");
    }

    @Override
    protected void doGet(final HttpServletRequest request, final HttpServletResponse response)
            throws ServletException, IOException {
        try {
            // Enable web sudo protection if needed and if the app we are running in supports it
            webSudoManager.willExecuteWebSudoRequest(request);
            final String[] pathParams = extractParams(request);
            final String typeId = pathParams[pathParams.length - 2];
            final String key = pathParams[pathParams.length - 1];

            publishAnalytics(typeId, key);

            final ListEntityLinksContext context = velocityContextFactory.buildListEntityLinksContext(request, typeId, key);

            // NOTE: any changes to the context parameters must be reflected in ListEntityLinksAction
            // It is recommended to use VelocityContextFactory instead (which will support both)
            render(TEMPLATE_PATH, createRenderContext(key, context), request, response);
        } catch (WebSudoSessionException wse) {
            webSudoManager.enforceWebSudoProtection(request, response);
        }
    }

    private void publishAnalytics(String typeId, String key) {
        eventPublisher.publish(new EntityLinksAdminViewEvent(typeId, key));
    }

    private Map<String, Object> createRenderContext(String key, ListEntityLinksContext context) {
        ImmutableMap.Builder<String, Object> mapBuilder = new ImmutableMap.Builder<>();
        mapBuilder.put("context", context);
        return internalHostApplication.getType().accept(new ApplicationTypeVisitor<ImmutableMap.Builder<String, Object>>() {
            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull BambooApplicationType type) {
                return mapBuilder.put("decorator", "atl.general");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull BitbucketApplicationType type) {
                return mapBuilder.put("projectKey", key)
                        .put("decorator", "bitbucket.project.settings")
                        .put("activeTab", "project-settings-entity-links");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull ConfluenceApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull CrowdApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull FishEyeCrucibleApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull GenericApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull JiraApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visit(@Nonnull RefAppApplicationType type) {
                return mapBuilder.put("decorator", "atl.admin");
            }

            @Override
            public ImmutableMap.Builder<String, Object> visitDefault(@Nonnull ApplicationType applicationType) {
                return mapBuilder.put("decorator", "atl.admin");
            }
        }).build();
    }

    private String[] extractParams(final HttpServletRequest request) {
        final String[] pathParams = StringUtils.split(request.getPathInfo(), '/');

        if (pathParams.length < 2) {
            throw new AbstractApplinksServlet.BadRequestException(messageFactory.newLocalizedMessage(
                    "Servlet URL should be of form /listEntityLinks/{entity-type}/{entity-key}"));
        }
        return pathParams;
    }
}