package com.atlassian.applinks.analytics;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.auth.AuthenticationProvider;
import com.atlassian.applinks.api.auth.types.BasicAuthenticationProvider;
import com.atlassian.applinks.api.auth.types.CorsAuthenticationProvider;
import com.atlassian.applinks.api.auth.types.OAuthAuthenticationProvider;
import com.atlassian.applinks.api.auth.types.TrustedAppsAuthenticationProvider;
import com.atlassian.applinks.api.auth.types.TwoLeggedOAuthAuthenticationProvider;
import com.atlassian.applinks.api.auth.types.TwoLeggedOAuthWithImpersonationAuthenticationProvider;
import com.atlassian.applinks.core.auth.AuthenticatorAccessor;
import com.atlassian.applinks.internal.event.StatusLoadedEvent;
import com.atlassian.applinks.internal.status.ApplinkStatus;
import com.atlassian.applinks.spi.auth.AuthenticationConfigurationManager;
import com.atlassian.applinks.spi.auth.AuthenticationProviderPluginModule;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Nonnull;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import static com.google.common.base.Preconditions.checkNotNull;

public class StatusAnalyticsPublisher {
    private final EventPublisher eventPublisher;
    private final AuthenticationConfigurationManager authenticationConfigurationManager;
    private final AuthenticatorAccessor authenticatorAccessor;

    @Autowired
    public StatusAnalyticsPublisher(final EventPublisher eventPublisher,
                                    final AuthenticationConfigurationManager manager,
                                    final AuthenticatorAccessor authenticatorAccessor) {
        this.eventPublisher = eventPublisher;
        this.authenticationConfigurationManager = manager;
        this.authenticatorAccessor = authenticatorAccessor;
    }

    @PostConstruct
    public void postInit() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void preDestroy() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void onStatusLoadedEvent(@Nonnull final StatusLoadedEvent event) {
        final ApplinkStatus status = checkNotNull(checkNotNull(event, "event").getStatus(), "status");
        ApplinksStatusEvent.Builder builder = new ApplinksStatusEvent.Builder(event.getStatus());
        findConfiguredProviderTypes(status.getLink().getId(), builder);
        eventPublisher.publish(builder.build());
    }

    private void findConfiguredProviderTypes(final ApplicationId id, final ApplinksStatusEvent.Builder builder) {
        for (AuthenticationProviderPluginModule module : authenticatorAccessor.getAllAuthenticationProviderPluginModules()) {
            final Class<? extends AuthenticationProvider> providerClass = module.getAuthenticationProviderClass();
            if (authenticationConfigurationManager.isConfigured(id, providerClass)) {
                if (BasicAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.basic = true;
                } else if (TrustedAppsAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.trusted = true;
                } else if (TwoLeggedOAuthAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.twoLo = true;
                } else if (TwoLeggedOAuthWithImpersonationAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.twoLoi = true;
                } else if (OAuthAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.threeLo = true;
                } else if (CorsAuthenticationProvider.class.isAssignableFrom(providerClass)) {
                    builder.cors = true;
                } else {
                    builder.other = true;
                }
            }
        }
    }
}
