(function (_, Backbone, templateNamespace, $) {

    AJS.$.namespace('Agent.Applinks.Table');

    /**
     * Backbone view for table row.
     */
    Agent.Applinks.TableRow = Backbone.View.extend({

        template: templateNamespace,

        initialize: function (options) {
            var item = options.model.toJSON();

            var unsupportedReasons = item.uiInstructions && item.uiInstructions.unsupportedReasons;

            this.$el.html(this.template.row({
                item: item,
                levels: options.levels,
                unsupportedReasons: unsupportedReasons
            }));
        },

        render: function () {
            return this.$el.find('tr');
        }

    });

    /**
     * Backbone view for inline form, emits save or cancel events.
     */
    Agent.Applinks.TableForm = Backbone.View.extend({

        template: templateNamespace,

        initialize: function (options) {
            this.model = options.model;

            this.$el.html(this.template.form({
                item: this.model.toJSON(),
                levels: options.levels,
                baseUrl: options.baseUrl
            }));

            this.$el
                // access level tracker
                .find('.aui-progress-tracker .level').on('click',
                    _.bind(this.onLevelChange, this)
                ).end()
                // form buttons
                .find('.cancel').on('click',_.bind(this.onCancel, this)).end()
                .find('button.save').on('click', _.bind(this.onSave, this)).end()
                // form dirty checking
                .find('input, input:hidden').on('change keyup',
                    _.bind(this.onFormChanged, this)
                )
        },

        /**
         * Change access level.
         */
        onLevelChange: function (event) {
            var $element = AJS.$(event.target);
            var $container = $element.parents('.aui-progress-tracker:first');
            var selectedClass = 'aui-progress-tracker-step-current';

            // stop current event
            event.preventDefault();
            event.stopPropagation();

            $container.find('li').removeClass(selectedClass);
            $container.siblings('input:hidden:first').val(
                $element.parents('li:first').addClass(selectedClass).data('value')
            ).trigger('change');

            // need to provide detailed warning when the levels are changed to L2 or L4
            this.$('.level1-warning')
                .toggleClass('hidden', !this.$('.level1.aui-progress-tracker-step-current:not(.previously-selected)').length);
            this.$('.level4-warning')
                .toggleClass('hidden', !this.$('.level4.aui-progress-tracker-step-current:not(.previously-selected)').length);
        },

        /**
         * When form content is changed, submit button is enabled and
         * then we unbind all events (which were registered).
         */
        onFormChanged: function() {
            this.$el.find('input, input:hidden').unbind();
            this.$el.find(':submit').removeAttr('disabled');
        },

        /**
         * Form was submitted.
         */
        onSave: function (event) {
            this.trigger('save',
                event, this.model.get('id')
            );
        },

        /**
         * Form dialog was canceled.
         */
        onCancel: function (event) {
            this.trigger('cancel',
                event, this.model.get('id')
            );
        },

        render: function () {
            return this.$el;
        }

    });

    /**
     * Dialog confirmation on delete.
     */
    Agent.Applinks.TableConfirmationDialog = Backbone.View.extend({

        template: templateNamespace,

        initialize: function (options) {
            // extend with options
            _.extend(this, options);

            this.$dialog = new AJS.Dialog({
                closeOnOutsideClick: true,
                height: "auto",
                width: 620
            });

            // dialog content
            this.$dialog.addHeader(AJS.I18n.getText('applinks.agent.dialog.confirmation.header'));
            this.$dialog.addPanel(null, this.template.confirmationDialog());

            // confirm button
            this.$dialog.addButton(
                AJS.I18n.getText('applinks.agent.action.confirm'),
                _.bind(this.onConfirm, this)
            );

            // cancel link
            this.$dialog.addLink(
                AJS.I18n.getText('applinks.agent.action.cancel'),
                _.bind(this.onCancel, this), '#'
            );
        },

        /**
         * Trigger confirm event and also call cancel logic.
         */
        onConfirm: function (dialog) {
            this.trigger('confirm', event, this.id);
            this.onCancel(dialog);
        },

        /**
         * On cancel hide dialog and remove element from DOM.
         */
        onCancel: function(dialog) {
            dialog.hide().remove();
        },

        /**
         * Show dialog.
         */
        show: function () {
            this.$dialog.show();
        }

    });

})(_, Backbone, agent.applinks.templates.table, AJS.$);
