(function (_, Backbone) {

    AJS.$.namespace('Agent.Applinks.Table');


    Agent.Applinks.TableModel = Backbone.Model.extend({

        initialize: function() {
            this.set({currentState: 'awaiting'})
            return Backbone.Model.prototype.initialize.call(this, arguments);
        },

        /**
         * Utility function to mark application link as primary.
         */
        makePrimary: function(options) {
            var model = this.toServerJSON();
            model.isPrimary = true;

            this.save(model, _.extend(options, {
                wait: true
            }));
        },

        getStatusTitle: function(status) {
            if (status == 1) {
                return AJS.I18n.getText("applinks.agent.status.mismatch");
            } else {
                return '';
            }
        },

        /**
         * Updates remote incoming & outgoing values.
         */
        setRemoteValues: function(data) {
            if (data.accessLevels && !_.isEmpty(data.accessLevels)) {
                // we need to do deep copy - so that model set functionality works correctly
                var accessLevels = AJS.$.extend(true, {}, this.get('accessLevels'));

                accessLevels.incoming.remote = data.accessLevels.incoming;
                accessLevels.outgoing.remote = data.accessLevels.outgoing;

                var incoming = accessLevels.incoming;
                var outgoing = accessLevels.outgoing;

                // check if they're in sync
                accessLevels.incoming.status = +(incoming.configured.accessLevel != outgoing.remote.configured.accessLevel);
                accessLevels.incoming.title = this.getStatusTitle(accessLevels.incoming.status);
                accessLevels.outgoing.status = +(outgoing.configured.accessLevel != incoming.remote.configured.accessLevel);
                accessLevels.outgoing.title = this.getStatusTitle(accessLevels.outgoing.status);
                this.set({ accessLevels: accessLevels });
            }

            if(data.remoteStatus && !_.isEmpty(data.remoteStatus)) {
                // we need to do deep copy - so that model set functionality works correctly
                var accessLevels = AJS.$.extend(true, {}, this.get('accessLevels'));

                accessLevels.remoteStatus = data.remoteStatus;

                accessLevels.remoteStatus.messages = data.remoteStatus.messages;

                if(data.link != null) {
                    var authenticationUrls = AJS.$.grep(data.link, function(link) {
                        if(link == null) {
                            return false;
                        }

                        return(link.rel === 'CREDENTIALS_REQUIRED');
                    });
                    if (authenticationUrls.length) {
                        accessLevels.remoteAuthorizationUrl = authenticationUrls[0].href;
                    }
                }

                this.set({ accessLevels: accessLevels });
            }
        },

        /**
         * Save model based on passed form values.
         */
        saveFormValues: function ($form, callbacks) {
            this.save(this._getFormObject($form), _.extend(callbacks, {
                wait: true
            }));
        },

        /**
         * Sets initial unknown (-1) remote configuration.
         */
        parse: function (response) {
            var unknownStatus = AJS.I18n.getText("applinks.agent.status.unknown");
            if (!response.accessLevels.incoming.remote) {
                response.accessLevels.incoming.status = -1;
                response.accessLevels.incoming.title = unknownStatus;
                response.accessLevels.incoming.remote = {
                    configured: 'UNKNOWN'
                };
            }
            if (!response.accessLevels.outgoing.remote) {
                response.accessLevels.outgoing.status = -1;
                response.accessLevels.outgoing.title = unknownStatus;
                response.accessLevels.outgoing.remote = {
                    configured: 'UNKNOWN'
                };
            }

            return response;
        },

        /**
         * Removes virtual attributes (remote & status) from JSON representation.
         */
        toServerJSON: function () {
            var model = this.toJSON();

            delete model.outgoing.status;
            delete model.outgoing.remote;

            delete model.incoming.status;
            delete model.incoming.remote;

            delete model.accessLevel;
            delete model.uiInstructions;

            return model;
        },

        /**
         * Serialize form to JSON object (name: value).
         */
        _getFormObject: function ($form) {
            var indexed = {};

            _.map($form.serializeArray(), _.bind(function (n, i) {
                this._setObjectByString(indexed, n['name'], n['value']);
            }, this));

            return indexed;
        },

        /**
         * Utility function which for object manipulation based on dot notation.
         */
        _setObjectByString: function (object, path, value) {
            var keys = path.split('.');

            for (var i = 0; i < (keys.length - 1); i++) {
                var key = keys[i];

                if (key in object) {
                    object = object[key];
                }
                else {
                    object[key] = {};
                    object = object[key];
                }
            }
            object[keys[keys.length - 1]] = value;
        }

    });


    /**
     * Table backbone collection, which is loaded based on URL.
     */
    Agent.Applinks.TableCollection = Backbone.Collection.extend({

        model: Agent.Applinks.TableModel,

        url: AJS.contextPath() + '/rest/applinks/3.0/config/local',

        fetch: function (options) {
            return Backbone.Collection.prototype.fetch.call(this, _.extend(options || {}));
        }

    });


})(_, Backbone, agent.applinks.templates.table);
