package com.atlassian.applinks.ui.auth;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

/**
 * Base class for servlet filter for urls that require at least certain privilege to access.
 *
 * @since 3.10
 */
public abstract class PowerUserFilter implements Filter
{
    private static final String LOGIN_SERVLET_PATH = "/plugins/servlet/applinks/login";

    protected final AdminUIAuthenticator uiAuthenticator;

    public PowerUserFilter(final AdminUIAuthenticator uiAuthenticator)
    {
        this.uiAuthenticator = uiAuthenticator;
    }

    public void doFilter(final ServletRequest servletRequest, final ServletResponse servletResponse,
                         final FilterChain filterChain) throws IOException, ServletException
    {
        if (!(servletRequest instanceof HttpServletRequest))
        {
            return;
        }

        final HttpServletRequest request = (HttpServletRequest) servletRequest;
        final HttpServletResponse response = (HttpServletResponse) servletResponse;

        if (!LOGIN_SERVLET_PATH.equals(request.getPathInfo()))
        {
            final String username = request.getParameter(AdminUIAuthenticator.ADMIN_USERNAME);
            final String password = request.getParameter(AdminUIAuthenticator.ADMIN_PASSWORD);

            if (!checkAccess(username, password, new ServletSessionHandler(request)))
            {
                handleAccessDenied(request, response);
                return;
            }
        }

        filterChain.doFilter(request, response);
    }

    /**
     * Handles access denied as decided by this filter.
     * By default, we simply redirect to a login page. Override this method if you want it to do something else.
     *
     * @param request request
     * @param response response
     * @throws IOException IOException
     */
    protected void handleAccessDenied(HttpServletRequest request, HttpServletResponse response) throws IOException
    {
        // By default, we redirect to login page so if user actually has the privilege, he will eventually get
        // redirected to the right page after the login.
        response.sendRedirect(
                new StringBuilder(request.getContextPath())
                        .append(LOGIN_SERVLET_PATH)
                        .append("?")
                        .append(AdminLoginServlet.ORIGINAL_URL).append("=").append(getOriginalUrl(request))
                        .append("&")
                        .append(AdminLoginServlet.FOR_ROLE).append("=").append(getForRole())
                        .toString()
        );
    }

    /***
     * Gets the required role which will be displayed in the admin login UI.
     *
     * @return required role.
     */
    abstract String getForRole();

    /**
     * Checks if the given username and password should be allowed to perform the operation.
     *
     * @param username username
     * @param password password
     * @param sessionHandler session handler
     * @return true if the user should be allowed to perform the operation.
     */
    abstract boolean checkAccess(String username, String password, AdminUIAuthenticator.SessionHandler sessionHandler);

    private String getOriginalUrl(final HttpServletRequest request)
            throws UnsupportedEncodingException
    {
        final String originalUrl = new StringBuilder(request.getContextPath())
                .append(request.getServletPath())
                .append(request.getPathInfo())
                .append(sanitiseQueryString(request))
                .toString();
        return URLEncoder.encode(originalUrl, "UTF-8");
    }

    private String sanitiseQueryString(final HttpServletRequest request)
    {
        String queryString = request.getQueryString();

        if (queryString == null)
        {
            queryString = "";
        }
        else
        {
            queryString = queryString.replaceAll("(&|^)al_(username|password)=[^&]*", "");
            if (queryString.length() > 0)
            {
                queryString = "?" + queryString;
            }
        }

        return queryString;
    }

    public void init(final FilterConfig filterConfig) throws ServletException
    {
        // sun rise
    }

    public void destroy()
    {
        // sun set
    }
}