package com.atlassian.applinks.ui.auth;

import com.atlassian.applinks.host.spi.InternalHostApplication;
import com.atlassian.sal.api.auth.AuthenticationListener;
import com.atlassian.sal.api.auth.Authenticator;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.user.UserManager;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.Serializable;


/**
 * Component for use in servlet filters and interceptors to limit AppLinks end-point access to administrators.
 */
public class AdminUIAuthenticator
{
    /**
     * Username request parameter
     */
    public static final String ADMIN_USERNAME = "al_username";
    /**
     * Password request parameter
     */
    public static final String ADMIN_PASSWORD = "al_password";

    private static final String ADMIN_SESSION_KEY = "al_auth";
    private static final String ADMIN = "admin";
    private static final String SYSADMIN = "sysadmin";

    private final UserManager userManager;
    private final AuthenticationListener authenticationListener;
    private final InternalHostApplication internalHostApplication;

    public AdminUIAuthenticator(final UserManager userManager, AuthenticationListener authenticationListener, InternalHostApplication internalHostApplication)
    {
        this.userManager = userManager;
        this.authenticationListener = authenticationListener;
        this.internalHostApplication = internalHostApplication;
    }

    /**
     * Check the supplied username and password, and set a session flag if valid.
     *
     * @param username       the username of the admin user to authenticate as
     * @param password       the password of the admin user to authenticate as
     * @param sessionHandler a {@link SessionHandler} for mutating the user's current session
     * @return <ul><li>true if the supplied username & password <strong>are not</strong> null and they match an admin user; or</li>
     *         <li>true if the supplied username & password <strong>are</strong> null, and the currently logged in user is an admin; or</li>
     *         <li>false otherwise.</li></ul>
     * @see #checkAdminUIAccessBySessionOrCurrentUser
     */
    public boolean checkAdminUIAccessBySessionOrPasswordAndActivateAdminSession(final String username, final String password, final SessionHandler sessionHandler)
    {
        if (isAdminSession(sessionHandler))
        {
            return true;
        }

        if (checkAdminUIAccessByPasswordOrCurrentUser(username, password))
        {
            sessionHandler.set(ADMIN_SESSION_KEY, ADMIN);
            return true;
        }

        return false;
    }

    /**
     * Check the supplied username and password, and set a session flag if valid.
     *
     * @param username       the username of the sysadmin user77 to authenticate as
     * @param password       the password of the sysadmin user to authenticate as
     * @param sessionHandler a {@link SessionHandler} for mutating the user's current session
     * @return <ul><li>true if the supplied username & password <strong>are not</strong> null and they match a sysadmin user; or</li>
     *         <li>true if the supplied username & password <strong>are</strong> null, and the currently logged in user is a sysadmin; or</li>
     *         <li>false otherwise.</li></ul>
     */
    public boolean checkSysadminUIAccessBySessionOrPasswordAndActivateSysadminSession(final String username, final String password, final SessionHandler sessionHandler)
    {
        if (isSysadminSession(sessionHandler))
        {
            return true;
        }

        if (checkSysadminUIAccessByPasswordOrCurrentUser(username, password))
        {
            sessionHandler.set(ADMIN_SESSION_KEY, SYSADMIN);
            return true;
        }

        return false;
    }

    /**
     * @param username the username of the admin user to authenticate as, can be null
     * @param password the password of the admin user to authenticate as, can be null
     * @return <ul><li>true if the supplied username & password <strong>are not</strong> null and they match an admin user; or</li>
     *         <li>true if the supplied username & password <strong>are</strong> null, and the currently logged in user is an admin; or</li>
     *         <li>false otherwise.</li></ul>
     */
    public boolean checkAdminUIAccessByPasswordOrCurrentUser(final String username, final String password)
    {
        if (username != null & password != null)
        {
            return userManager.authenticate(username, password) && isAdmin(username);
        }
        else
        {
            return isCurrentUserAdmin();
        }
    }

    /**
     * @param username the username of the admin user to authenticate as, can be null
     * @param password the password of the admin user to authenticate as, can be null
     * @return <ul><li>true if the supplied username & password <strong>are not</strong> null and they match an admin user; or</li>
     *         <li>true if the supplied username & password <strong>are</strong> null, and the currently logged in user is an admin; or</li>
     *         <li>false otherwise.</li></ul>
     */
    public boolean checkSysadminUIAccessByPasswordOrCurrentUser(final String username, final String password)
    {
        if (username != null & password != null)
        {
            return userManager.authenticate(username, password) && isSysadmin(username);
        }
        else
        {
            return isCurrentUserSysadmin();
        }
    }

    public boolean checkAdminUIAccessBySessionOrCurrentUser(final HttpServletRequest request)
    {
        String username = userManager.getRemoteUsername();
        return isAdminSession(request) || isAdmin(username);
    }

    public boolean checkSysadminUIAccessBySessionOrCurrentUser(final HttpServletRequest request)
    {
        String username = userManager.getRemoteUsername();
        return isSysadminSession(request) || isSysadmin(username);
    }

    public boolean isCurrentUserAdmin()
    {
        return isAdmin(userManager.getRemoteUsername());
    }

    public boolean isCurrentUserSysadmin()
    {
        return isSysadmin(userManager.getRemoteUsername());
    }

    public Result logInAsPowerUser(final String username, final String password,
                                   HttpServletRequest request, HttpServletResponse response)
    {
        if (username != null && password != null)
        {
            if (userManager.authenticate(username, password))
            {
                if (isAdmin(username))
                {
                    if (isSysadmin(username))
                    {
                        request.getSession().setAttribute(ADMIN_SESSION_KEY, SYSADMIN);
                    }
                    else
                    {
                        request.getSession().setAttribute(ADMIN_SESSION_KEY, ADMIN);
                    }

                    return returnPowerUserLoginSuccess(username, request, response);
                }
                else
                {
                    return returnNotPowerUser(username);
                }
            }
        }

        return returnLoginFailed();
    }

    private Result returnLoginFailed()
    {
        return new Result(false, new Message()
        {
            public String getKey()
            {
                return "applinks.admin.login.auth.failed";
            }

            public Serializable[] getArguments()
            {
                return new Serializable[0];
            }
        });
    }

    private Result returnNotPowerUser(final String username)
    {
        return new Result(false, new Message()
        {
            public String getKey()
            {
                return "applinks.admin.login.auth.authorization.failed";
            }

            public Serializable[] getArguments()
            {
                return new Serializable[]{ username, internalHostApplication.getName() };
            }
        });
    }

    private Result returnPowerUserLoginSuccess(String username, HttpServletRequest request, HttpServletResponse response)
    {
        // HACK: the only way we could actually log into the container (as of SAL 2.4.0)
        Message message = new Message()
        {
            public String getKey()
            {
                return "Successfully authenticated";
            }

            public Serializable[] getArguments()
            {
                return null;
            }
        };
        Authenticator.Result result = new Authenticator.Result.Success(message, userManager.resolve(username));
        // This is the only SAL-provided method of logging into the instance.
        authenticationListener.authenticationSuccess(result, request, response);
        return SUCCESS;
    }

    private static final Result SUCCESS = new Result(true);

    static class Result
    {
        private final boolean success;
        private final Message message;

        public Result(final boolean success)
        {
            this(success, null);
        }

        Result(final boolean success, final Message message)
        {
            this.success = success;
            this.message = message;
        }

        public boolean success()
        {
            return success;
        }

        public Message getMessage()
        {
            return message;
        }
    }

    private boolean isAdmin(final String username)
    {
        return username != null && (userManager.isAdmin(username) || userManager.isSystemAdmin(username));
    }

    private boolean isSysadmin(final String username)
    {
        return username != null && userManager.isSystemAdmin(username);
    }

    private boolean isAdminSession(HttpServletRequest request)
    {
        return isAdminSession(new ServletSessionHandler(request));
    }

    private boolean isAdminSession(SessionHandler sessionHandler)
    {
        return ADMIN.equals(sessionHandler.get(ADMIN_SESSION_KEY)) || SYSADMIN.equals(sessionHandler.get(ADMIN_SESSION_KEY));
    }

    private boolean isSysadminSession(HttpServletRequest request)
    {
        return isSysadminSession(new ServletSessionHandler(request));
    }

    private boolean isSysadminSession(SessionHandler sessionHandler)
    {
        return SYSADMIN.equals(sessionHandler.get(ADMIN_SESSION_KEY));
    }

    public static interface SessionHandler
    {
        void set(String key, Object value);

        Object get(String key);
    }
}
