package com.atlassian.applinks.core.v2.rest;

import java.net.URI;
import java.net.URISyntaxException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLink;
import com.atlassian.applinks.api.ApplicationType;
import com.atlassian.applinks.api.TypeNotInstalledException;
import com.atlassian.applinks.api.auth.AuthenticationProvider;
import com.atlassian.applinks.api.auth.types.TwoLeggedOAuthWithImpersonationAuthenticationProvider;
import com.atlassian.applinks.application.generic.GenericApplicationTypeImpl;
import com.atlassian.applinks.core.InternalTypeAccessor;
import com.atlassian.applinks.core.auth.oauth.OAuthHelper;
import com.atlassian.applinks.core.auth.oauth.ServiceProviderStoreService;
import com.atlassian.applinks.core.rest.auth.AdminApplicationLinksInterceptor;
import com.atlassian.applinks.core.rest.context.ContextInterceptor;
import com.atlassian.applinks.core.rest.model.ApplicationLinkAuthenticationEntity;
import com.atlassian.applinks.core.rest.model.ApplicationLinkEntity;
import com.atlassian.applinks.core.rest.model.AuthenticationProviderEntity;
import com.atlassian.applinks.core.rest.model.AuthenticationProviderEntityListEntity;
import com.atlassian.applinks.core.rest.model.ConsumerEntity;
import com.atlassian.applinks.core.rest.model.ConsumerEntityListEntity;
import com.atlassian.applinks.spi.application.ApplicationIdUtil;
import com.atlassian.applinks.spi.auth.AuthenticationConfigurationManager;
import com.atlassian.applinks.spi.auth.AuthenticationProviderPluginModule;
import com.atlassian.applinks.spi.link.ApplicationLinkDetails;
import com.atlassian.applinks.spi.link.MutableApplicationLink;
import com.atlassian.applinks.spi.link.MutatingApplicationLinkService;
import com.atlassian.applinks.spi.manifest.ManifestNotFoundException;
import com.atlassian.applinks.spi.manifest.ManifestRetriever;
import com.atlassian.oauth.Consumer;
import com.atlassian.oauth.consumer.ConsumerService;
import com.atlassian.oauth.util.RSAKeys;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugins.rest.common.Link;
import com.atlassian.plugins.rest.common.interceptor.InterceptorChain;
import com.atlassian.plugins.rest.common.util.RestUrlBuilder;
import com.atlassian.sal.api.message.I18nResolver;
import com.atlassian.sal.api.net.RequestFactory;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.websudo.WebSudoRequired;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.sun.jersey.spi.resource.Singleton;

import org.apache.commons.lang.StringUtils;

import static com.atlassian.applinks.core.rest.util.RestUtil.badRequest;
import static com.atlassian.applinks.core.rest.util.RestUtil.created;
import static com.atlassian.applinks.core.rest.util.RestUtil.forbidden;
import static com.atlassian.applinks.core.rest.util.RestUtil.notFound;
import static com.atlassian.applinks.core.rest.util.RestUtil.ok;
import static com.atlassian.applinks.core.rest.util.RestUtil.serverError;
import static com.atlassian.applinks.core.rest.util.RestUtil.updated;

/**
 * V2.0 of applicationLink REST resource, add resources for managing authentication.
 * @since 4.0
 */
@Path(ApplicationLinkResource.CONTEXT)
@Consumes({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
@Singleton
@WebSudoRequired
@InterceptorChain ({ ContextInterceptor.class, AdminApplicationLinksInterceptor.class })
public class ApplicationLinkResource extends com.atlassian.applinks.core.v1.rest.ApplicationLinkResource
{
    private final PluginAccessor pluginAccessor;
    private final AuthenticationConfigurationManager authenticationConfigurationManager;
    private final ServiceProviderStoreService serviceProviderStoreService;
    private final ConsumerService consumerService;

    public ApplicationLinkResource(final MutatingApplicationLinkService applicationLinkService,
            final I18nResolver i18nResolver,
            final InternalTypeAccessor typeAccessor,
            final ManifestRetriever manifestRetriever,
            final RestUrlBuilder restUrlBuilder,
            final RequestFactory requestFactory,
            final UserManager userManager,
            final PluginAccessor pluginAccessor,
            final AuthenticationConfigurationManager authenticationConfigurationManager,
            final ServiceProviderStoreService serviceProviderStoreService,
            final ConsumerService consumerService)
    {
        super(applicationLinkService,
                i18nResolver,
                typeAccessor,
                manifestRetriever,
                restUrlBuilder,
                requestFactory,
                userManager);
        this.pluginAccessor = pluginAccessor;
        this.authenticationConfigurationManager = authenticationConfigurationManager;
        this.serviceProviderStoreService = serviceProviderStoreService;
        this.consumerService = consumerService;
    }

    /**
     * Creates an application link.
     *
     * @param applicationLink The new details to set.
     * @return 'badRequest' if the target application is not reachable or if the name already exists; 'forbidden' if
     * the user performing the operation does not have sysadmin privilege; 'created' in case of success.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     */
    @PUT
    public Response addApplicationLink(final ApplicationLinkEntity applicationLink)
            throws TypeNotInstalledException
    {
        final ApplicationType applicationType = typeAccessor.loadApplicationType(applicationLink.getTypeId());
        if (applicationType == null)
        {
            LOG.warn(String.format("Couldn't load type %s for application link. Type is not installed?", applicationLink.getTypeId()));
            throw new TypeNotInstalledException(applicationLink.getTypeId().get(), applicationLink.getName(), applicationLink.getRpcUrl());
        }

        if(!(applicationType instanceof GenericApplicationTypeImpl))
        {
            // TODO is this still needed. why bother?
            // for atlassian apps We get the manifest, just to check the application is up and running
            try
            {
                manifestRetriever.getManifest(applicationLink.getRpcUrl(), applicationType);
            }
            catch(ManifestNotFoundException e)
            {
                return badRequest(i18nResolver.getText("applinks.error.url.application.not.reachable", applicationLink.getRpcUrl()));
            }
        }

        ApplicationId applicationId = applicationLink.getId();

        if(applicationType instanceof GenericApplicationTypeImpl
                || applicationId == null)
        {
            // historically for generic application links the id is created internally from the url.
            // so follow precedent and ignore anything passed in and create on from the rpcUrl
            applicationId = ApplicationIdUtil.generate(applicationLink.getRpcUrl());
        }

        // Fetch the original & mutable ApplicationLink record.
        final MutableApplicationLink existing = applicationLinkService.getApplicationLink(applicationId);

        // If the id already exists we can't create the a duplicate link.
        if (applicationLinkService.getApplicationLink(applicationId) != null)
        {
            // report the error based on the RPC URL because either
            // a) its a generic link in which case the id is effectively a hash of the RPC URL
            // b) its an Atlassian link in which case the id comes form the manifest, which comes vis the RPC URL.
            // and c) the ID is meaningless to the user.
            return badRequest(i18nResolver.getText("applinks.error.duplicate.url", applicationLink.getRpcUrl()));
        }

        applicationLinkService.addApplicationLink(applicationId, applicationType, applicationLink.getDetails());

        return created(createSelfLinkFor(applicationId));
    }

    /**
     * Updates the application link but not for updating RPC URL of the link.
     *
     * @param id the Application ID.
     * @param applicationLink The new details to set.
     * @return 'badRequest' if the target application is not reachable or if the name already exists; 'forbidden' if
     * the user performing the operation does not have sysadmin privilege; 'updated' in case of success.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     */
    @POST
    @Path ("{id}")
    public Response updateApplicationLink(@PathParam ("id") final String id, final ApplicationLinkEntity applicationLink)
            throws TypeNotInstalledException
    {
        // We get the manifest, just to check the application is up and running
        final ApplicationType applicationType = typeAccessor.loadApplicationType(applicationLink.getTypeId());
        if (applicationType == null)
        {
            LOG.warn(String.format("Couldn't load type %s for application link. Type is not installed?", applicationLink.getTypeId()));
            throw new TypeNotInstalledException(applicationLink.getTypeId().get(), applicationLink.getName(), applicationLink.getRpcUrl());
        }

        // Fetch the original & mutable ApplicationLink record.
        ApplicationId applicationId = new ApplicationId(id);
        final MutableApplicationLink existing = applicationLinkService.getApplicationLink(applicationId);

        // If it doesn't exist, we can't update the link
        if (existing == null)
        {
            return badRequest(i18nResolver.getText("applinks.notfound", applicationLink.getName()));
        }

        // only a sysadmin can update a system link.
        if (existing.isSystem() && !userManager.isSystemAdmin(userManager.getRemoteUsername()))
        {
            return forbidden(i18nResolver.getText("applinks.error.only.sysadmin.operation"));
        }

        ApplicationLinkDetails linkDetails = applicationLink.getDetails();
        // We need to check the name doesn't already exist.
        if (applicationLinkService.isNameInUse(linkDetails.getName(), applicationId))
        {
            return badRequest(i18nResolver.getText("applinks.error.duplicate.name", applicationLink.getName()));
        }

        // the relocation functionality should be used to change rpcurl instead of this sneaky way.
        if (!existing.getRpcUrl().equals(linkDetails.getRpcUrl()))
        {
            return badRequest(i18nResolver.getText("applinks.error.cannot.update.rpcurl"));
        }

        existing.update(linkDetails);
        return updated(createSelfLinkFor(applicationLink.getId()));
    }

    /**
     * Gets a description of the Authentication configuration of the ApplicationLink defined by the unique id.
     * @param id the id of the ApplicationLink
     * @return 'notfound' if the ApplicationLink cannot be found
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     * @throws URISyntaxException if an invalid url is used when creating the "self" link in AuthenticationProviderEntities or ConsumerEntities.
     */
    @GET
    @Path ("{id}/authentication")
    public Response getAuthentication(@PathParam ("id") final String id)
            throws TypeNotInstalledException, URISyntaxException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        // get the providers configured for this link
        final List<AuthenticationProviderEntity> configuredAuthProviders = getConfiguredProviders(applicationLink);

        Iterable<Consumer> consumers = findConsumers(applicationLink, configuredAuthProviders);


        List<ConsumerEntity> consumerEntities = Lists.newArrayList();
        for(Consumer consumer : consumers)
        {
            consumerEntities.add(new ConsumerEntity(Link.self(new URI(CONTEXT + "/" + id + "/authentication/consumer")),consumer));
        }

        return ok(new ApplicationLinkAuthenticationEntity(Link.self(new URI(CONTEXT + "/" + id + "/authentication")), consumerEntities, configuredAuthProviders));
    }

    /**
     * Gets a list of the providers configured for the ApplicationLink specified by the unique id.
     * @param id the id of the ApplicationLink
     * @return 'notfound' if the specified ApplicationLink cannot be found. 'servererror' if there is a problem generating the AuthenticationProviderEntities, 'ok' for success.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     */
    @GET
    @Path ("{id}/authentication/provider")
    public Response getAuthenticationProvider(@PathParam ("id") final String id)
            throws TypeNotInstalledException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        final List<AuthenticationProviderEntity> configuredAuthProviders;
        try
        {
            configuredAuthProviders = getConfiguredProviders(applicationLink);
        }
        catch (URISyntaxException e)
        {
            return serverError(i18nResolver.getText("applinks.authenticationproviders.notfound", id));
        }

        return ok(new AuthenticationProviderEntityListEntity(configuredAuthProviders));
    }

    /**
     * Gets the configuration of the specified Authentication Provider for the the specified ApplicationLink.
     * @param id the id of the ApplicationLink
     * @param provider the fully qualified type of the provider type, e.g. 'com.atlassian.applinks.api.auth.types.OAuthAuthenticationProvider'
     * @return 'notfound' if the specified ApplicationLink cannot be found. 'servererror' if there is a problem generating the AuthenticationProviderEntities, 'ok' for success.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     */
    @GET
    @Path ("{id}/authentication/provider/{provider}")
    public Response getAuthenticationProvider(@PathParam ("id") final String id, @PathParam ("provider") final String provider)
            throws TypeNotInstalledException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        final List<AuthenticationProviderEntity> configuredAuthProviders;
        try
        {
            configuredAuthProviders = getConfiguredProviders(applicationLink, getProviderPredicate(provider));
        }
        catch (URISyntaxException e)
        {
            return serverError(i18nResolver.getText("applinks.authenticationproviders.notfound", id));
        }

        return ok(new AuthenticationProviderEntityListEntity(configuredAuthProviders));
    }

    /**
     * Registers the specified AuthenticationProvider against the specified ApplicationLink
     * @param id the unique id of the ApplicationId.
     * @param authenticationProviderEntity the AuthenticationProvider definition.
     * @return 'notfound' if the ApplicationLink cannot be found, 'badrequest' if the specified AuthenticationProvider is not recognized, 'created' otherwise.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     * @throws URISyntaxException if an invalid url is used when creating the "self" link in AuthenticationProviderEntities.
     */
    @PUT
    @Path ("{id}/authentication/provider")
    public Response putAuthenticationProvider(@PathParam ("id") final String id, final AuthenticationProviderEntity authenticationProviderEntity)
            throws TypeNotInstalledException, URISyntaxException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        try
        {
            final Class providerClass = Class.forName(authenticationProviderEntity.getProvider());

            // only Sys Admin can set 2LOi
            if(TwoLeggedOAuthWithImpersonationAuthenticationProvider.class.equals(providerClass)
                    && !userManager.isSystemAdmin(userManager.getRemoteUsername()))
            {
                return badRequest(i18nResolver.getText("applinks.authentication.provider.2loi.only.available.to.sysadmin", id, authenticationProviderEntity.getProvider()));
            }

            authenticationConfigurationManager.registerProvider(applicationLink.getId(), providerClass, authenticationProviderEntity.getConfig());
            return created(Link.self(new URI(CONTEXT + "/" + id + "/authentication/" + authenticationProviderEntity.getProvider())));
        }
        catch (ClassNotFoundException e)
        {
            return badRequest(i18nResolver.getText("applinks.authentication.provider.type.not.recognized", id, authenticationProviderEntity.getProvider()));
        }
    }

    /**
     * Get the configuration of the Consumer for the specified ApplicationLink.
     * @param id the unique id of the ApplicationId.
     * @return 'notfound' if the ApplicationLink cannot be found, 'badrequest' if the specified AuthenticationProvider is not recognized, 'created' otherwise.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     * @throws URISyntaxException if an invalid url is used when creating the "self" link in AuthenticationProviderEntities or ConsumerEntities
     */
    @GET
    @Path ("{id}/authentication/consumer")
    public Response getConsumer(@PathParam ("id") final String id)
            throws TypeNotInstalledException, URISyntaxException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        // get the providers configured for this link
        final List<AuthenticationProviderEntity> configuredAuthProviders = getConfiguredProviders(applicationLink);

        Iterable<Consumer> consumers = findConsumers(applicationLink, configuredAuthProviders);

        if(!consumers.iterator().hasNext() && applicationLink.getType() instanceof GenericApplicationTypeImpl
            && configuredAuthProviders.size() == 0)
        {
            return notFound(i18nResolver.getText("applinks.generic.consumer.needs.authenticationprovider", id));
        }

        if(!consumers.iterator().hasNext())
        {
            return notFound(i18nResolver.getText("applinks.consumer.notfound", id));
        }

        List<ConsumerEntity> consumerEntities = Lists.newArrayList();
        for(Consumer consumer : consumers)
        {
            consumerEntities.add(new ConsumerEntity(Link.self(new URI(CONTEXT + "/" + id + "/authentication/consumer")), consumer));
        }

        return ok(new ConsumerEntityListEntity(consumerEntities));
    }

    /**
     * Add as a Consumer and its configuration to the specified ApplicationLink
     * @param id the unique id of the ApplicationLink
     * @param autoConfigure a flag allowing a shortcut of Consumer configuration by using the existing ApplicationLink to
     *  retrieve the Consumer information from the remote application.
     * @param consumerEntity the Consumer definition.
     * @return 'notfound' if the ApplicationLink can't be found, 'servererror' if autoconfigure has been requested but fails,
     * 'badrequest' if there is a problem with the submitted Consumer configuration and 'created' on success.
     * @throws com.atlassian.applinks.api.TypeNotInstalledException if applicationLink.getTypeId() refers to a non-installed type
     * @throws URISyntaxException if an invalid url is used when creating the "self" link in ConsumerEntities
     */
    @PUT
    @Path ("{id}/authentication/consumer")
    public Response putConsumer(@PathParam ("id") final String id, @QueryParam ("autoConfigure") final Boolean autoConfigure, final ConsumerEntity consumerEntity)
            throws TypeNotInstalledException, URISyntaxException
    {
        ApplicationLink applicationLink = this.findApplicationLink(id);

        if (applicationLink == null)
        {
            return notFound(i18nResolver.getText("applinks.notfound", id));
        }

        // auto configuration from the existing application link
        if(autoConfigure != null && autoConfigure)
        {
            try
            {
                // ask the remote app for Consumer info via the existing applink url
                Consumer consumer = OAuthHelper.fetchConsumerInformation(applicationLink);
                // update the consumer information to reflect the 2LO settings passed in.
                Consumer updatedConsumer = new Consumer.InstanceBuilder(consumer.getKey())
                        .name(consumer.getName())
                        .description(consumer.getDescription())
                        .publicKey(consumer.getPublicKey())
                        .signatureMethod(consumer.getSignatureMethod())
                        .callback(consumer.getCallback())
                        .twoLOAllowed(consumerEntity.isTwoLOAllowed())
                        .executingTwoLOUser(consumerEntity.getExecutingTwoLOUser())
                        .twoLOImpersonationAllowed(consumerEntity.isTwoLOImpersonationAllowed())
                        .build();

                // store the Consumer
                serviceProviderStoreService.addConsumer(updatedConsumer, applicationLink);
            }
            catch (ResponseException e)
            {
                return serverError(i18nResolver.getText("applinks.consumer.autoconfigure.consumerInfo.notfound"));
            }

            return created(Link.self(new URI(CONTEXT + "/" + id + "/authentication/consumer")));
        }


        // manual configuration

        Consumer consumer;
        if(applicationLink.getType() instanceof GenericApplicationTypeImpl)
        {
            // 3rd Party Consumers

            List<String> errors = validate3rdPartyConsumer(consumerEntity);
            if(errors.size() > 0)
            {
                return badRequest(errors.toArray(new String[errors.size()]));
            }

            if( consumerEntity.isOutgoing())
            {
                // create an outgoing Consumer
                this.add3rdPartyOutgoingConsumer(consumerEntity);
            }
            else
            {
                // create an incoming Consumer
                try
                {
                    consumer = this.createBasicConsumer(consumerEntity, applicationLink);
                    serviceProviderStoreService.addConsumer(consumer, applicationLink);
                }
                catch (NoSuchAlgorithmException e)
                {
                    return badRequest(i18nResolver.getText("applinks.invalid.consumer.publickey", id));
                }
                catch (InvalidKeySpecException e)
                {
                    return badRequest(i18nResolver.getText("applinks.invalid.consumer.publickey", id));
                }
            }
        }
        else
        {
            List<String> errors = validateAtlassianConsumer(consumerEntity);
            if(errors.size() > 0)
            {
                return badRequest(errors.toArray(new String[errors.size()]));
            }

            try
            {
                consumer = this.createBasicConsumer(consumerEntity, applicationLink);

                // update the consumer information to reflect the 2LO settings passed in.
                Consumer updatedConsumer = new Consumer.InstanceBuilder(consumer.getKey())
                        .name(consumer.getName())
                        .description(consumer.getDescription())
                        .publicKey(consumer.getPublicKey())
                        .signatureMethod(consumer.getSignatureMethod())
                        .callback(consumer.getCallback())
                        .twoLOAllowed(consumerEntity.isTwoLOAllowed())
                        .executingTwoLOUser(consumerEntity.getExecutingTwoLOUser())
                        .twoLOImpersonationAllowed(consumerEntity.isTwoLOImpersonationAllowed())
                        .build();

                // store the Consumer
                serviceProviderStoreService.addConsumer(updatedConsumer, applicationLink);

            }
            catch (NoSuchAlgorithmException e)
            {
                return badRequest(i18nResolver.getText("applinks.invalid.consumer.publickey", id));
            }
            catch (InvalidKeySpecException e)
            {
                return badRequest(i18nResolver.getText("applinks.invalid.consumer.publickey", id));
            }
        }

        return created(Link.self(new URI(CONTEXT + "/" + id + "/authentication/consumer")));
    }

    /**
     * Add an Outgoing Consumer for a 3rdParty link.
     */
    private Consumer add3rdPartyOutgoingConsumer(ConsumerEntity consumerEntity)
    {
        final Consumer consumer = Consumer.key(consumerEntity.getKey())
                .name(consumerEntity.getName())
                .signatureMethod(Consumer.SignatureMethod.HMAC_SHA1)
                .description(consumerEntity.getDescription())
                .build();

        consumerService.add(consumer.getName(), consumer, consumerEntity.getSharedSecret());

        return consumer;
    }

    /**
     * Create a basic Consumer for the specified ApplicationLink
     */
    private Consumer createBasicConsumer(ConsumerEntity consumerEntity, ApplicationLink applicationLink)
            throws InvalidKeySpecException, NoSuchAlgorithmException
    {
        return Consumer.key(consumerEntity.getKey())
                .name(consumerEntity.getName())
                .publicKey(RSAKeys.fromPemEncodingToPublicKey(consumerEntity.getPublicKey()))
                .description(consumerEntity.getDescription())
                .callback(consumerEntity.getCallback())
                .build();
    }

    /**
     * check the pre-requisites for creating a 3rdparty Consumer
     */
    private List<String> validate3rdPartyConsumer(ConsumerEntity consumerEntity)
    {
        List<String> errors = Lists.newArrayList();

        // check minimum parameters see AddServiceProviderManuallyServlet.save
        if(StringUtils.isEmpty(consumerEntity.getKey()))
        {
            errors.add(i18nResolver.getText("auth.oauth.config.consumer.serviceprovider.key.is.required"));
        }

        if( consumerEntity.isOutgoing())
        {
            if(StringUtils.isEmpty(consumerEntity.getName()))
            {
                errors.add(i18nResolver.getText("auth.oauth.config.consumer.serviceprovider.name.is.required"));
            }

            if(StringUtils.isEmpty(consumerEntity.getSharedSecret()))
            {
                errors.add(i18nResolver.getText("auth.oauth.config.consumer.serviceprovider.shared.secret.is.required"));
            }
        }
        else
        {
            if(StringUtils.isEmpty(consumerEntity.getPublicKey()))
            {
                errors.add(i18nResolver.getText("auth.oauth.config.serviceprovider.missing.public.key"));
            }
        }

        return errors;
    }

    /**
     * check the pre-requisites for creating an atlassian Consumer
     */
    private List<String> validateAtlassianConsumer(ConsumerEntity consumerEntity)
    {
        List<String> errors = Lists.newArrayList();

        if(StringUtils.isEmpty(consumerEntity.getKey()))
        {
            errors.add(i18nResolver.getText("auth.oauth.config.consumer.serviceprovider.key.is.required"));
        }

        if(StringUtils.isEmpty(consumerEntity.getName()))
        {
            errors.add(i18nResolver.getText("auth.oauth.config.consumer.serviceprovider.name.is.required"));
        }

        if(StringUtils.isEmpty(consumerEntity.getPublicKey()))
        {
            errors.add(i18nResolver.getText("applinks.consumer.publickey.required"));
        }
        return errors;
    }

    /**
     * Get a Predicate to use to find AuthenticationProviders
     */
    private Predicate<AuthenticationProviderPluginModule> getProviderPredicate(final String provider)
    {
        return new Predicate<AuthenticationProviderPluginModule>()
        {
            public boolean apply(final AuthenticationProviderPluginModule input)
            {
                return input.getAuthenticationProviderClass().getName().equals(provider);
            }
        };
    }

    /**
     * Get filtered Configured providers for the current ApplicationLink.
     */
    private List<AuthenticationProviderEntity> getConfiguredProviders(ApplicationLink applicationLink, Predicate<AuthenticationProviderPluginModule> predicate)
            throws URISyntaxException
    {
        return getConfiguredProviders(applicationLink, Iterables.filter(pluginAccessor.getEnabledModulesByClass(AuthenticationProviderPluginModule.class), predicate));
    }

    /**
     * Get unfiltered Configured providers for the current ApplicationLink.
     */
    private List<AuthenticationProviderEntity> getConfiguredProviders(ApplicationLink applicationLink)
            throws URISyntaxException
    {
        return getConfiguredProviders(applicationLink, pluginAccessor.getEnabledModulesByClass(AuthenticationProviderPluginModule.class));
    }

    /**
     * Get Configured providers for the current ApplicationLink.
     */
    private List<AuthenticationProviderEntity> getConfiguredProviders(ApplicationLink applicationLink, Iterable<AuthenticationProviderPluginModule> pluginModules)
            throws URISyntaxException
    {
        // get the providers configured for this link
        final List<AuthenticationProviderEntity> configuredAuthProviders = new ArrayList<AuthenticationProviderEntity>();
        for (AuthenticationProviderPluginModule authenticationProviderPluginModule : pluginModules)
        {
            final AuthenticationProvider authenticationProvider = authenticationProviderPluginModule.getAuthenticationProvider(applicationLink);
            if (authenticationProvider != null)
            {
                Map<String, String> config = authenticationConfigurationManager.getConfiguration(applicationLink.getId(),authenticationProviderPluginModule.getAuthenticationProviderClass());
                configuredAuthProviders.add(new AuthenticationProviderEntity(
                        Link.self(new URI(CONTEXT + "/" + applicationLink.getId().toString() + "/authentication/provider")),
                        authenticationProviderPluginModule.getClass().getName(),
                        authenticationProviderPluginModule.getAuthenticationProviderClass().getName(),
                        config));
            }
        }
        return configuredAuthProviders;
    }

    /**
     * Find the ApplicationLink by its id.
     */
    private ApplicationLink findApplicationLink(final String id) throws TypeNotInstalledException
    {
        ApplicationId applicationId;
        try
        {
            applicationId = new ApplicationId(id);
        }
        catch(IllegalArgumentException e)
        {
            return null;
        }

        return applicationLinkService.getApplicationLink(applicationId);
    }

    /**
     * find an ApplicationLinks's Consumer
     */
    private Iterable<Consumer> findConsumers(ApplicationLink applicationLink, List<AuthenticationProviderEntity> configuredAuthProviders)
    {
        List<Consumer> consumers = Lists.newArrayList();
        Consumer consumer = serviceProviderStoreService.getConsumer(applicationLink);

        if(consumer != null)
        {
            // incoming consumer
            consumers.add(consumer);
        }

        // for generic applicationlinks the Consumer is stored in the OAuth ConsumerService using the consumerkey as the key
        // the consumerkey is stored as a property of the Authenticationprovider.
        if(applicationLink.getType() instanceof GenericApplicationTypeImpl)
        {
            for(AuthenticationProviderEntity entity : configuredAuthProviders)
            {
                if(applicationLink.getType() instanceof GenericApplicationTypeImpl)
                {
                    final String consumerKey = entity.getConfig().get("consumerKey.outbound");
                    if(StringUtils.isEmpty(consumerKey))
                    {
                        continue;
                    }

                    Consumer genericOutGoingConsumer = consumerService.getConsumerByKey(consumerKey);
                    if(genericOutGoingConsumer != null)
                    {
                        // outgoing consumer
                        consumers.add(genericOutGoingConsumer);
                    }
                }
            }
        }

        return consumers;
    }
}
