package com.atlassian.applinks.core.util;

import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;

/**
 * Redirect helper responsible for determining if a redirect should be followed
 * and where the next location is.
 *
 * @since 3.11.0
 */
public class RedirectHelper
{
    public static final int MAX_REDIRECTS = 3;

    private static final Logger log = LoggerFactory.getLogger(RedirectHelper.class);

    private int redirects = 0;

    public String getNextRedirectLocation(final Response response) throws ResponseException
    {
        final String location = response.getHeader("location");
        redirects++;
        return location;
    }

    public boolean responseShouldRedirect(final Response response)
    {
         return isRedirectStatusCode(response) && hasLocation(response) && notExceededMaximumRedirects();
    }

    private boolean isRedirectStatusCode(final Response response)
    {
        return (response.getStatusCode() >= 300 && response.getStatusCode() < 400);
    }

    private boolean hasLocation(final Response response)
    {
        final String location = response.getHeader("location");
        if (isBlank(location)) {
            log.warn("HTTP response returned redirect code " + response.getStatusCode() + " but did not provide a location header");
        }

        return isNotBlank(location);
    }

    private boolean notExceededMaximumRedirects()
    {
        if (redirects >= MAX_REDIRECTS)
        {
            log.warn("Maximum of " + MAX_REDIRECTS + " redirects reached.");
            return false;
        }

        return true;
    }
}
