package com.atlassian.applinks.core.auth.trusted;

import com.atlassian.applinks.api.ApplicationLinkResponseHandler;
import com.atlassian.applinks.core.auth.AbstractApplicationLinkRequest;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.net.ResponseHandler;
import com.atlassian.sal.api.net.ReturningResponseHandler;
import com.atlassian.security.auth.trustedapps.CurrentApplication;
import com.atlassian.security.auth.trustedapps.TrustedApplicationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of a trusted application links request. This will ensure that
 * requests are signed correctly and executed with the a response handler.
 *
 * @since 3.11.0
 */
public class TrustedRequest extends AbstractApplicationLinkRequest
{
    private static final Logger log = LoggerFactory.getLogger(TrustedRequest.class);

    private final CurrentApplication currentApplication;
    private final String username;

    public TrustedRequest(final String url, final Request wrappedRequest, final CurrentApplication currentApplication,
                          final String username)
    {
        super(url, wrappedRequest);
        this.currentApplication = currentApplication;
        this.username = username;
    }

    public <R> R execute(final ApplicationLinkResponseHandler<R> applicationLinkResponseHandler) throws ResponseException
    {
        signRequest();
        return wrappedRequest.execute(new TrustedApplinksResponseHandler<R>(applicationLinkResponseHandler, this, followRedirects));
    }

    public void execute(final ResponseHandler responseHandler) throws ResponseException
    {
        signRequest();
        wrappedRequest.execute(new TrustedResponseHandler(responseHandler, this, followRedirects));
    }

    public <R> R executeAndReturn(final ReturningResponseHandler<Response, R> returningResponseHandler) throws ResponseException
    {
        signRequest();
        return wrappedRequest.executeAndReturn(new TrustedApplinksReturningResponseHandler<Response, R>(returningResponseHandler, this, followRedirects));
    }


    @Override
    public void signRequest() throws ResponseException
    {
        signRequest(unsignedRequest());
    }

    public void signRequest(final com.atlassian.security.auth.trustedapps.request.TrustedRequest unsignedRequest)
    {
        if (log.isDebugEnabled())
        {
            log.debug("signRequest - signing request for url:" + url);
        }
        TrustedApplicationUtils.addRequestParameters(currentApplication.encode(username, url), unsignedRequest);
    }

    public com.atlassian.security.auth.trustedapps.request.TrustedRequest unsignedRequest()
    {
        return new com.atlassian.security.auth.trustedapps.request.TrustedRequest()
        {
            public void addRequestParameter(final String name, final String value)
            {
                wrappedRequest.setHeader(name, value);
            }
        };
    }

}