package com.atlassian.applinks.core.auth.oauth;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLinkResponseHandler;
import com.atlassian.applinks.core.auth.AbstractApplicationLinkRequest;
import com.atlassian.oauth.ServiceProvider;
import com.atlassian.oauth.consumer.ConsumerService;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.net.ResponseHandler;
import com.atlassian.sal.api.net.ReturningResponseHandler;
import net.oauth.OAuth;
import net.oauth.OAuthMessage;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * @since 3.10
 */
public abstract class OAuthRequest extends AbstractApplicationLinkRequest
{
    protected final MethodType methodType;
    protected final ApplicationId applicationId;
    protected final ServiceProvider serviceProvider;
    protected final ConsumerService consumerService;

    public OAuthRequest(final String url, final MethodType methodType, final Request wrappedRequest,
                        final ApplicationId applicationId, final ServiceProvider serviceProvider,
                        final ConsumerService consumerService) {
        super(url, wrappedRequest);
        this.methodType = methodType;
        this.applicationId = applicationId;
        this.serviceProvider = serviceProvider;
        this.consumerService = consumerService;
    }

    public <R> R execute(final ApplicationLinkResponseHandler<R> applicationLinkResponseHandler) throws ResponseException
    {
        signRequest();
        return wrappedRequest.execute(new OAuthApplinksResponseHandler<R>(applicationLinkResponseHandler, this, applicationId, followRedirects));
    }

    public void execute(final ResponseHandler responseHandler) throws ResponseException
    {
        signRequest();
        wrappedRequest.execute(new OAuthResponseHandler(responseHandler, wrappedRequest, applicationId, followRedirects));
    }

    public <R> R executeAndReturn(final ReturningResponseHandler<Response, R> returningResponseHandler)
            throws ResponseException {
        signRequest();
        return wrappedRequest.executeAndReturn(
                new OAuthApplinksReturningResponseHandler<Response, R>(returningResponseHandler, wrappedRequest, applicationId, followRedirects));
    }

    protected void signRequest() throws ResponseException
    {
        final com.atlassian.oauth.Request oAuthRequest = createUnsignedRequest();
        final com.atlassian.oauth.Request signedRequest = consumerService.sign(oAuthRequest, serviceProvider);
        final OAuthMessage oAuthMessage = OAuthHelper.asOAuthMessage(signedRequest);
        try
        {
            wrappedRequest.setHeader("Authorization", oAuthMessage.getAuthorizationHeader(null));
        }
        catch (IOException e)
        {
            throw new ResponseException("Unable to generate OAuth Authorization request header.", e);
        }
    }

    /**
     * @return OAuth request before being signed, all the parameters must be available and ready to be signed.
     */
    protected abstract com.atlassian.oauth.Request createUnsignedRequest();

    protected List<com.atlassian.oauth.Request.Parameter> toOAuthParameters(final String accesstoken)
    {
        final List<com.atlassian.oauth.Request.Parameter> parameters = new ArrayList<com.atlassian.oauth.Request.Parameter>();
        parameters.add(new com.atlassian.oauth.Request.Parameter(OAuth.OAUTH_TOKEN, accesstoken));
        for (final String parameterName : this.parameters.keySet())
        {
            final List<String> values = this.parameters.get(parameterName);
            for (final String value : values)
            {
                parameters.add(new com.atlassian.oauth.Request.Parameter(parameterName, value));
            }
        }
        return parameters;
    }

    protected static com.atlassian.oauth.Request.HttpMethod toOAuthMethodType(final Request.MethodType methodType)
    {
        com.atlassian.oauth.Request.HttpMethod method = com.atlassian.oauth.Request.HttpMethod.GET;

        if (methodType == Request.MethodType.GET)
        {
            method = com.atlassian.oauth.Request.HttpMethod.GET;
        }else if (methodType == Request.MethodType.POST)
        {
            method = com.atlassian.oauth.Request.HttpMethod.POST;
        }else if (methodType == Request.MethodType.PUT)
        {
            method = com.atlassian.oauth.Request.HttpMethod.PUT;
        }else if (methodType == Request.MethodType.DELETE)
        {
            method = com.atlassian.oauth.Request.HttpMethod.DELETE;
        }
        return method;
    }
}