package com.atlassian.applinks.core.auth.oauth;

import com.atlassian.applinks.api.ApplicationId;
import com.atlassian.applinks.api.ApplicationLinkRequest;
import com.atlassian.applinks.core.auth.AbstractApplicationLinkResponseHandler;
import com.atlassian.sal.api.net.Response;
import com.google.common.collect.ImmutableSet;
import net.oauth.OAuth;
import net.oauth.OAuthMessage;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Set;

/**
 * Base class for OAuth redirecting application link response handlers.
 *
 * @since 3.11.0
 */
public class OAuthRedirectingApplicationLinkResponseHandler extends AbstractApplicationLinkResponseHandler
{
    private static final Logger log = LoggerFactory.getLogger(OAuthRedirectingApplicationLinkResponseHandler.class);

    protected static final Set<String> TOKEN_PROBLEMS = ImmutableSet.of(
            // See: http://wiki.oauth.net/ProblemReporting
            OAuth.Problems.TOKEN_EXPIRED,
            OAuth.Problems.TOKEN_REJECTED,
            OAuth.Problems.TOKEN_REVOKED
    );

    protected final ConsumerTokenStoreService consumerTokenStoreService;
    protected final ApplicationId applicationId;
    protected final String username;

    // Used to indicate a problem with the token and to redirect to credentials required.
    protected boolean hasTokenProblems = false;

    public OAuthRedirectingApplicationLinkResponseHandler(ApplicationLinkRequest wrappedRequest, ConsumerTokenStoreService consumerTokenStoreService,
                                                          ApplicationId applicationId, String username, boolean followRedirects)
    {
        super(wrappedRequest, followRedirects);
        this.consumerTokenStoreService = consumerTokenStoreService;
        this.username = username;
        this.applicationId = applicationId;
    }

    protected void checkForOAuthProblemAndRemoveConsumerTokenIfNecessary(Response response)
    {
        final String value = response.getHeaders().get("WWW-Authenticate");
        if (!StringUtils.isBlank(value))
        {
            for (OAuth.Parameter parameter : OAuthMessage.decodeAuthorization(value))
            {
                if ("oauth_problem".equals(parameter.getKey()))
                {
                    log.debug("OAuth request rejected by peer.\n" +
                            "Our OAuth request header: Authorization: " + wrappedRequest.getHeaders().get("Authorization") + "\n" +
                            "Full OAuth response header: WWW-Authenticate: " + value);

                    if (OAuth.Problems.TIMESTAMP_REFUSED.equals(parameter.getValue()))
                    {
                        log.warn("Peer rejected the timestamp on our OAuth request. " +
                                "This might be due to a replay attack, but it's more " +
                                "likely our system clock is not synchronized with the " +
                                "server's clock. " +
                                "You may turn on debug logging to log the full contents " +
                                "of the OAuth response headers.");
                    }

                    // if the consumer token store is not provided, we just can't automatically delete it.
                    if (consumerTokenStoreService != null)
                    {
                        if (TOKEN_PROBLEMS.contains(parameter.getValue()))
                        {
                            try
                            {
                                consumerTokenStoreService.removeConsumerToken(applicationId, username);
                            }
                            catch (RuntimeException e)
                            {
                                log.error("Failed to delete consumer token for user '" + username + "'.", e);
                            }
                            hasTokenProblems = true;
                        }
                    }
                }
            }
        }
    }
}
