package com.atlassian.application.host.plugin;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.plugin.PluginAccessor;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import io.atlassian.fugue.Option;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Set;

import static com.atlassian.application.host.util.Arguments.checkArgumentNotNull;
import static io.atlassian.fugue.Iterables.findFirst;

/**
 * Provides access to the {@link com.atlassian.application.host.plugin.PluginApplicationMetaData} installed in the
 * platform. It reads the metadata exposed by plugins via
 * {@link com.atlassian.application.host.plugin.ApplicationMetaDataModuleDescriptor}s
 *
 * @since v1.0
 */
public final class DefaultPluginApplicationMetaDataManager implements PluginApplicationMetaDataManager
{
    private static final Logger log = LoggerFactory.getLogger(DefaultPluginApplicationMetaDataManager.class);

    private final PluginAccessor accessor;

    /**
     * Create a new manager that will use the passed {@link com.atlassian.plugin.PluginAccessor} to find all the
     * installed {@code PluginApplicationMetaData}.
     *
     * @param accessor the accessor used to find all the installed {@code PluginApplicationMetaData}
     */
    public DefaultPluginApplicationMetaDataManager(final PluginAccessor accessor)
    {
        this.accessor = checkArgumentNotNull(accessor, "accessor");
    }

    @Override
    public Iterable<PluginApplicationMetaData> getApplications()
    {
        final List<ApplicationMetaDataModuleDescriptor> moduleDescriptors =
                accessor.getEnabledModuleDescriptorsByClass(ApplicationMetaDataModuleDescriptor.class);

        final ImmutableList.Builder<PluginApplicationMetaData> uniqueModuleDescriptors = ImmutableList.builder();
        final Set<ApplicationKey> keys = Sets.newHashSet();
        for (ApplicationMetaDataModuleDescriptor moduleDescriptor: moduleDescriptors)
        {
            final PluginApplicationMetaData module = moduleDescriptor.getModule();
            if (keys.add(module.getKey()))
            {
                uniqueModuleDescriptors.add(module);
            }
            else
            {
                log.debug(String.format("Application descriptor with id '%s' has a duplicate.",
                        module.getKey()));
            }
        }
        return uniqueModuleDescriptors.build();
    }

    @Override
    public Option<PluginApplicationMetaData> getApplication(final ApplicationKey key)
    {
        checkArgumentNotNull(key, "key");

        return findFirst(getApplications(), new Predicate<PluginApplicationMetaData>()
        {
            @Override
            public boolean apply(final PluginApplicationMetaData input)
            {
                return input.getKey().equals(key);
            }
        });
    }
}
