package com.atlassian.application.host.plugin;

import com.atlassian.application.api.ApplicationAccess;
import com.atlassian.application.api.ApplicationKey;
import com.atlassian.application.api.ApplicationPlugin;
import com.atlassian.application.api.PluginApplication;
import com.atlassian.application.host.ApplicationConfigurationManager;
import com.atlassian.application.host.i18n.I18nResolver;
import com.atlassian.sal.api.license.SingleProductLicenseDetailsView;
import com.google.common.base.Supplier;
import io.atlassian.fugue.Option;

import java.net.URI;
import java.time.ZonedDateTime;

import static com.atlassian.application.host.util.Arguments.checkArgumentNotNull;

/**
 * An implementation of the {@link com.atlassian.application.api.PluginApplication} interface. It uses the passed
 * {@link PluginApplicationMetaData} to implement most of its functionality.
 *
 * @since v1.0
 */
public final class PluginApplicationImpl implements PluginApplication
{
    private final PluginApplicationMetaData metaData;
    private final I18nResolver resolver;
    private final Supplier<Option<SingleProductLicenseDetailsView>> license;
    private final Supplier<ApplicationAccess> access;
    private final ApplicationConfigurationManager appConfigManager;

    /**
     * Create a new {@code PluginApplicationImpl}.
     *
     * @param metaData the metaData used query application information.
     * @param resolver the i18n resolver to use when translating text.
     * @param license the license for the application
     * @param access the access for the application.
     * @param appConfigManager the application configuration manager.
     */
    public PluginApplicationImpl(final PluginApplicationMetaData metaData,
            final I18nResolver resolver,
            final Supplier<Option<SingleProductLicenseDetailsView>> license,
            final Supplier<ApplicationAccess> access,
            final ApplicationConfigurationManager appConfigManager)
    {
        this.license = checkArgumentNotNull(license, "license");
        this.metaData = checkArgumentNotNull(metaData, "metaData");
        this.resolver = checkArgumentNotNull(resolver, "resolver");
        this.access = checkArgumentNotNull(access, "access");
        this.appConfigManager = checkArgumentNotNull(appConfigManager, "appConfigManager");
    }

    @Override
    public ApplicationKey getKey()
    {
        return metaData.getKey();
    }

    @Override
    public String getName()
    {
        return metaData.getName();
    }

    @Override
    public String getVersion()
    {
        return metaData.getVersion();
    }

    @Override
    public String getDescription()
    {
        return resolver.getText(metaData.getDescriptionKey());
    }

    @Override
    public String getUserCountDescription(final Option<Integer> count)
    {
        return resolver.getText(metaData.getUserCountKey(), count.getOrElse(-1));
    }

    @Override
    public Option<URI> getConfigurationURI()
    {
        return metaData.getConfigurationURI();
    }

    @Override
    public Option<URI> getPostInstallURI()
    {
        return metaData.getPostInstallURI();
    }

    @Override
    public Option<URI> getPostUpdateURI()
    {
        return metaData.getPostUpdateURI();
    }

    @Override
    public Option<URI> getProductHelpServerSpaceURI()
    {
        return metaData.getProductHelpServerSpaceURI();
    }

    @Override
    public Option<URI> getProductHelpCloudSpaceURI()
    {
        return metaData.getProductHelpCloudSpaceURI();
    }

    @Override
    public ZonedDateTime buildZonedDate()
    {
        return metaData.buildZonedDate();
    }

    @Override
    public Option<SingleProductLicenseDetailsView> getLicense()
    {
        return license.get();
    }

    @Override
    public ApplicationAccess getAccess()
    {
        return access.get();
    }

    @Override
    public String getDefaultGroup()
    {
        return metaData.getDefaultGroup();
    }

    @Override
    public void clearConfiguration()
    {
        appConfigManager.clearConfiguration(getKey());
    }

    @Override
    public String getDefinitionModuleKey()
    {
        return metaData.getDefinitionModuleKey();
    }

    @Override
    public Iterable<ApplicationPlugin> getPlugins()
    {
        return metaData.getPlugins();
    }

    @Override
    public ApplicationPlugin getPrimaryPlugin()
    {
        return metaData.getPrimaryPlugin();
    }

    @Override
    public Iterable<ApplicationPlugin> getApplicationPlugins()
    {
        return metaData.getApplicationPlugins();
    }

    @Override
    public Iterable<ApplicationPlugin> getUtilityPlugins()
    {
        return metaData.getUtilityPlugins();
    }

    @Override
    public String toString()
    {
        return "Plugin Application: " + getKey();
    }
}
