package com.atlassian.application.host.plugin;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.impl.AbstractDelegatingPlugin;
import com.atlassian.plugin.osgi.factory.OsgiPlugin;
import com.google.common.base.Function;
import org.joda.time.DateTime;
import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import static com.atlassian.fugue.Option.none;
import static com.atlassian.fugue.Option.option;

/**
 * Parses the AMPS build date embedded in the {@code Atlassian-Build-Date} OSGi manifest header.
 *
 * @since v1.0
 */
@Immutable
@ThreadSafe
final class AmpsBuildDateParser implements Function<Plugin, Option<DateTime>>
{
    private static final Logger log = LoggerFactory.getLogger(AmpsBuildDateParser.class);

    private static final String BUILD_DATE_ATTRIBUTE = "Atlassian-Build-Date";

    /**
     * Creates a new date parser.
     */
    AmpsBuildDateParser()
    {
    }

    @Override
    public Option<DateTime> apply(@Nullable final Plugin plugin)
    {
        if (plugin == null)
        {
            return none();
        }
        else
        {
            for (Bundle bundle : getPluginBundle(plugin))
            {
                final Object value = bundle.getHeaders().get(BUILD_DATE_ATTRIBUTE);
                if (value != null)
                {
                    final Option<DateTime> buildDate = PluginDateTimeUtils.fromPluginString(value.toString());
                    if (buildDate.isEmpty())
                    {
                        log.debug("Plugin with key \"{}\" has invalid Atlassian-Build-Date of \"{}\".",
                                plugin.getKey(), value);
                    }
                    return buildDate;
                }
            }
            return none();
        }
    }

    private static Option<Bundle> getPluginBundle(final Plugin plugin)
    {
        if (plugin instanceof OsgiPlugin)
        {
            return option(((OsgiPlugin) plugin).getBundle());
        }
        else if (plugin instanceof AbstractDelegatingPlugin)
        {
            return getPluginBundle(((AbstractDelegatingPlugin) plugin).getDelegate());
        }
        return none();
    }
}
