package com.atlassian.analytics.client.hash.reader;

import com.atlassian.analytics.client.s3.AnalyticsS3Client;
import com.google.common.base.Optional;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;


/**
 * Retrieves hashing instructions to be used by the {@link com.atlassian.analytics.client.hash.AnalyticsEmailHasher}
 * from an S3 bucket.
 */
public class RemoteHashingInstructionsReader implements HashingInstructionsReader {
    private static final Logger LOG = LoggerFactory.getLogger(RemoteHashingInstructionsReader.class);

    private static final String ANALYTICS_CONFIG_S3_BUCKET_NAME = "btf-analytics";
    private static final String ANALYTICS_CONFIG_S3_KEY_PREFIX = "config/";

    @Override
    public Optional<String> readInstructions(String instructionsKey) {
        try {
            final Optional<InputStream> remoteListObject = getRemoteListObject(instructionsKey);
            if (remoteListObject.isPresent()) {
                return Optional.fromNullable(StringUtils.trim(IOUtils.toString(remoteListObject.get())));
            } else {
                LOG.warn("Unable to read remote instructions with key '{}'.", instructionsKey);
            }
        } catch (IOException e) {
            LOG.warn("Unable to read remote instructions with key '{}'. Enable debug logging for more info.", instructionsKey);

            if (LOG.isDebugEnabled()) {
                LOG.debug("Unable to read remote instructions with key '{}'", instructionsKey, e);
            }
        }
        return Optional.absent();
    }

    private Optional<InputStream> getRemoteListObject(final String objectName) {
        return Optional.fromNullable(new AnalyticsS3Client().getS3ObjectInputStream(ANALYTICS_CONFIG_S3_BUCKET_NAME, ANALYTICS_CONFIG_S3_KEY_PREFIX + objectName));
    }
}
