package com.atlassian.analytics.client.eventfilter.whitelist;

import com.atlassian.analytics.api.events.MauEvent;
import com.atlassian.analytics.client.eventfilter.AllowedWordFilter;
import com.atlassian.analytics.client.eventfilter.parser.JsonListParser;
import com.atlassian.analytics.client.eventfilter.reader.RemoteListReader;
import com.atlassian.analytics.client.logger.EventAnonymizer;
import com.atlassian.analytics.client.properties.AnalyticsPropertyService;
import com.atlassian.analytics.client.serialize.IsMauEventAvailable;
import com.atlassian.analytics.event.RawEvent;
import com.atlassian.sal.api.features.DarkFeatureManager;
import com.atlassian.util.concurrent.LazyReference;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.regex.Pattern;

import static com.atlassian.analytics.client.eventfilter.whitelist.WhitelistFilter.EventKey.eventKey;

/**
 * Filter that checks whether an event or event attribute is whitelisted,
 * used to determine whether it should be passed through the allowed word dictionary or not.
 */
public class WhitelistFilter
{
    private static final Logger LOG = LoggerFactory.getLogger(WhitelistFilter.class);
    private static final Pattern EXPERIMENT_PATTERN = Pattern.compile("^grow-?\\d+[^a-zA-Z0-9].+$", Pattern.CASE_INSENSITIVE);

    // Grow prefixed events are auto whitelisted, useful in BTF instances
    static final String DARK_FEATURE_AUTO_WHITELIST_GROW_EVENTS_KEY = "com.atlassian.analytics.auto.whitelist.grow.events";

    private final AnalyticsPropertyService analyticsPropertyService;

    //Three caches above do not use cache loader because they need to use least common deniminator API
    //from guava 11.0.2 (Confluence) and guava 18.0 (JIRA)
    private final LoadingCache<EventKey, Boolean> hashedCache = CacheBuilder
            .newBuilder()
            .build(new CacheLoader<EventKey, Boolean>()
            {
                @Override
                public Boolean load(EventKey key) throws Exception
                {
                    return globalWhitelist.shouldAttributeBeHashed(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeHashed(key.event, key.property);
                }
            });

    private final LoadingCache<EventKey, Boolean> dictionaryFilteredCache = CacheBuilder
            .newBuilder()
            .build(new CacheLoader<EventKey, Boolean>()
            {
                @Override
                public Boolean load(EventKey key) throws Exception
                {
                    return globalWhitelist.shouldAttributeBeDictionaryFiltered(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeDictionaryFiltered(key.event, key.property);
                }
            });

    private final LoadingCache<EventKey, Boolean> whitelistedCache = CacheBuilder
            .newBuilder()
            .build(new CacheLoader<EventKey, Boolean>()
            {
                @Override
                public Boolean load(EventKey key) throws Exception
                {
                    return globalWhitelist.shouldAttributeBeWhitelisted(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeWhitelisted(key.event, key.property);
                }
            });

    private final AllowedWordFilter allowedWordFilter;
    private final WhitelistCollector whitelistCollector;
    private final EventAnonymizer eventAnonymizer;
    private final Whitelist globalWhitelist;
    private final AggregatedWhitelist pluginWhitelists;
    private final DarkFeatureManager darkFeatureManager;
    private final LazyReference<Boolean> isMauEventAvailable = new IsMauEventAvailable();

    private Boolean autoWhitelistGrow = null;

    public WhitelistFilter(final AnalyticsPropertyService analyticsPropertyService, final AllowedWordFilter allowedWordFilter,
                           final WhitelistCollector whitelistCollector, final EventAnonymizer eventAnonymizer,
                           final DarkFeatureManager darkFeatureManager)
    {
        this.analyticsPropertyService = analyticsPropertyService;
        this.allowedWordFilter = allowedWordFilter;
        this.whitelistCollector = whitelistCollector;
        this.eventAnonymizer = eventAnonymizer;
        this.globalWhitelist = Whitelist.createEmptyWhitelist();
        this.pluginWhitelists = AggregatedWhitelist.createEmptyAggregate();
        this.darkFeatureManager = darkFeatureManager;
    }

    private void initialiseWhitelists()
    {
        // Collect all externally defined whitelists
        final List<Whitelist> externalWhitelists = whitelistCollector.collectExternalWhitelists();
        final Whitelist globalWhitelist = getGlobalWhitelist(externalWhitelists);
        if (globalWhitelist != null)
        {
            // The global whitelist will be read and treated differently
            this.globalWhitelist.initialiseFrom(globalWhitelist);
            externalWhitelists.remove(globalWhitelist);
        }
        this.pluginWhitelists.initialiseFrom(externalWhitelists);

        this.hashedCache.invalidateAll();
        this.dictionaryFilteredCache.invalidateAll();
        this.whitelistedCache.invalidateAll();
    }

    private Whitelist getGlobalWhitelist(final List<Whitelist> externalWhitelists)
    {
        for (Whitelist whitelist : externalWhitelists)
        {
            if (whitelist.isGlobalWhitelist())
            {
                return whitelist;
            }
        }
        return null;
    }

    private String getProductName()
    {
        return analyticsPropertyService.getDisplayName().toLowerCase();
    }

    public static String getListName(final String appName)
    {
        return "whitelist_" + appName + ".json";
    }

    public boolean isEventWhitelisted(final RawEvent event, final boolean isOnDemand)
    {
        final String eventName = event.getName();
        return isEventAlwaysWhitelisted(eventName, isOnDemand) || isEventWhitelisted(eventName);
    }

    private boolean isEventAlwaysWhitelisted(final String eventName, final boolean isOnDemand)
    {
        // If the dark feature is enabled, whitelist "grow"-prefixed events automatically, even in BTF
        if (autoWhitelistGrow == null)
        {
            autoWhitelistGrow = darkFeatureManager.isFeatureEnabledForCurrentUser(DARK_FEATURE_AUTO_WHITELIST_GROW_EVENTS_KEY);
        }

        // Always whitelist experiment events
        final boolean isGrowthExperimentAllowed = (isOnDemand || autoWhitelistGrow) && EXPERIMENT_PATTERN.matcher(eventName).find();
        // MAU events are always allowed through and we already hash properties on these events to make them safe.
        final boolean isMauEvent = isMauEventAvailable.get() && MauEvent.EVENT_NAME.equals(eventName);
        return isGrowthExperimentAllowed || isMauEvent;
    }

    public Map<String, Object> applyWhitelistToEvent(final String eventName, final Map<String, Object> rawProperties,
                                                     final boolean isOnDemand)
    {
        final Map<String, Object> processedProperties = Maps.newHashMap(rawProperties);

        // If the event is automatically excluded, and not contained in the whitelist (e.g. experiment event or mau), do nothing
        if (isEventAlwaysWhitelisted(eventName, isOnDemand) && !isEventWhitelisted(eventName))
        {
            return processedProperties;
        }

        final Set<String> removeProperties = new HashSet<>();
        for (Map.Entry<String, Object> property : rawProperties.entrySet())
        {
            final String propertyName = property.getKey();

            final Object originalPropertyValue = property.getValue();
            if (originalPropertyValue == null || originalPropertyValue instanceof Number || originalPropertyValue instanceof Boolean) {
                continue;
            } else if (originalPropertyValue instanceof Enum) {
                processedProperties.put(propertyName, ((Enum) originalPropertyValue).name());
                continue;
            }

            final String propertyValue = String.valueOf(originalPropertyValue);

            // If the attribute value is in a provided allowed list, don't apply any filtering
            if (isAttributeNotInAllowedList(eventName, propertyName, propertyValue))
            {
                // Check if event attribute should be filtered - if none of the conditions are met, the attribute is left untouched
                // If we are in OnDemand, we don't hash or dictionary filter attributes - this needs to be its own conditional to allow those attributes to be left untouched
                if (shouldAttributeBeHashed(eventName, propertyName))
                {
                    hashProperty(processedProperties, propertyName, propertyValue, isOnDemand);
                }
                else if (shouldAttributeBeDictionaryFiltered(eventName, propertyName))
                {
                    // these used to be filtered via the common terms dictionary - now just pruned, unless they are
                    // numeric or hash-like
                    applyAllowedWordsFiltering(eventName, processedProperties, propertyName, propertyValue, isOnDemand);
                }
                else if (!shouldAttributeBeWhitelisted(eventName, propertyName))
                {
                    // If none of the filter conditions are met, this property is not analytics safe and should be removed!
                    removeProperties.add(propertyName);
                }
            }
        }
        for (String propertyName : removeProperties)
        {
            processedProperties.remove(propertyName);
        }
        return processedProperties;
    }

    private boolean isEventWhitelisted(final String eventName)
    {
        return globalWhitelist.isEventWhitelisted(eventName) ||
               pluginWhitelists.isEventWhitelisted(eventName);
    }

    private boolean isAttributeNotInAllowedList(String eventName, String propertyName, String propertyValue)
    {
        return !pluginWhitelists.isAttributeValueInAllowedList(eventName, propertyName, propertyValue);
    }

    private boolean shouldAttributeBeHashed(final String eventName, final String propertyName)
    {
        final EventKey key = eventKey(eventName, propertyName);
        try
        {
            return hashedCache.get(key, new Callable<Boolean>()
            {
                @Override
                public Boolean call() throws Exception
                {
                    return globalWhitelist.shouldAttributeBeHashed(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeHashed(key.event, key.property);
                }
            });
        }
        catch (ExecutionException e)
        {
            //any exception in callable will be wrapped into ExecutionException
            throw Throwables.propagate(e);
        }
    }

    private boolean shouldAttributeBeDictionaryFiltered(final String eventName, final String propertyName)
    {
        final EventKey key = eventKey(eventName, propertyName);
        try
        {
            return dictionaryFilteredCache.get(key, new Callable<Boolean>()
            {
                @Override
                public Boolean call() throws Exception
                {
                    return globalWhitelist.shouldAttributeBeDictionaryFiltered(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeDictionaryFiltered(key.event, key.property);
                }
            });
        }
        catch (ExecutionException e)
        {
            //any exception in callable will be wrapped into ExecutionException
            throw Throwables.propagate(e);
        }
    }

    private boolean shouldAttributeBeWhitelisted(final String eventName, final String propertyName)
    {
        final EventKey key = eventKey(eventName, propertyName);
        try
        {
            return whitelistedCache.get(key, new Callable<Boolean>()
            {
                @Override
                public Boolean call() throws Exception
                {
                    return globalWhitelist.shouldAttributeBeWhitelisted(key.event, key.property) ||
                            pluginWhitelists.shouldAttributeBeWhitelisted(key.event, key.property);
                }
            });
        }
        catch (ExecutionException e)
        {
            //any exception in callable will be wrapped into ExecutionException
            throw Throwables.propagate(e);
        }
    }

    private void hashProperty(final Map<String, Object> properties, final String propertyName,
                              final String propertyValue, final boolean isOnDemand)
    {
        // Only apply MD5 hashing in BTF instances
        if (!isOnDemand)
        {
            properties.put(propertyName, eventAnonymizer.hashEventProperty(propertyValue));
        }
    }

    private void applyAllowedWordsFiltering(String eventName, final Map<String, Object> properties, final String propertyName,
                                            final String propertyValue, final boolean isOnDemand)
    {
        // Only apply dictionary filtering in BTF instances
        if (!isOnDemand)
        {
            final String processedPropertyValue = allowedWordFilter.processAllowedWords(propertyValue);
            if (LOG.isDebugEnabled() && processedPropertyValue.isEmpty()) {
                LOG.debug("Discarded value for property {} of event {}", propertyName, eventName);
            }
            properties.put(propertyName, processedPropertyValue);
        }
    }

    public void readRemoteList()
    {
        try
        {
            final JsonListParser jsonListParser = new JsonListParser(new RemoteListReader());
            globalWhitelist.initialiseFrom(jsonListParser.readJsonFilterList(getListName(getProductName())));
        }
        catch (Exception e)
        {
            LOG.debug("Couldn't read the remote whitelist, using the local whitelist for now - exception message: " + e.getMessage());
        }
    }

    public void collectExternalWhitelists()
    {
        initialiseWhitelists();
    }

    public Whitelist getGlobalWhitelist()
    {
        return globalWhitelist;
    }

    public List<Whitelist> getPluginWhitelists()
    {
        return pluginWhitelists.getWhitelists();
    }

    public List<Whitelist.WhitelistBean> toWhitelistBeans()
    {
        List<Whitelist.WhitelistBean> whitelistBeans = new ArrayList<Whitelist.WhitelistBean>();
        whitelistBeans.add(globalWhitelist.toWhitelistBean());
        whitelistBeans.addAll(pluginWhitelists.toWhitelistBeans());

        return whitelistBeans;
    }

    static class EventKey {
        final String event;
        final String property;

        static EventKey eventKey(String event, String property)
        {
            return new EventKey(event, property);
        }

        private EventKey(String event, String property)
        {
            this.event = event;
            this.property = property;
        }

        public boolean equals(Object o) {
            if(o == null) {
                return false;
            } else if(this == o) {
                return true;
            } else if(!(o instanceof EventKey)) {
                return false;
            } else {
                EventKey that = (EventKey)o;
                return this.event.equals(that.event) && this.property.equals(that.property);
            }
        }

        public int hashCode() {
            int eh = this.event.hashCode();
            int ph = this.property.hashCode();
            return 31*eh + ph;
        }
    }
}
