package com.atlassian.analytics.client.eventfilter.parser;

import com.atlassian.analytics.client.eventfilter.reader.FilterListReader;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.Set;

/**
 * Read an event filter file which contains event names to allow/disallow.
 * <p></p>
 * File format should be as follows:
 * <p></p>
 * <p>
 * # This is a comment
 * eventName1
 * eventName2
 * </p>
 */
public class SimpleListParser
{
    private static final Logger LOG = LoggerFactory.getLogger(SimpleListParser.class);

    private final FilterListReader filterListReader;

    public SimpleListParser(FilterListReader filterListReader)
    {
        this.filterListReader = filterListReader;
    }

    public Set<String> readSimpleFilterList(final String filterListName)
    {
        Set<String> filteredEvents = new HashSet<String>();
        InputStream listInputStream = filterListReader.readFilterList(filterListName);

        if (listInputStream != null)
        {
            LOG.debug("Reading filter list resource, {}", filterListName);
            BufferedReader br = new BufferedReader(new InputStreamReader(listInputStream));
            String line;
            try
            {
                while ((line = br.readLine()) != null)
                {
                    final String eventName = line.trim();
                    if (!eventName.startsWith("#") && StringUtils.isNotBlank(eventName))
                    {
                        filteredEvents.add(eventName);
                    }
                }
            }
            catch (IOException e)
            {
                LOG.debug("Failed reading filter list resource, {}", filterListName);
            }
        }
        else
        {
            LOG.debug("Couldn't find filter list resource, {}", filterListName);
            return null;
        }

        return filteredEvents;
    }
}
