package com.atlassian.analytics.client.browser;

import com.atlassian.analytics.client.TimeKeeper;
import com.atlassian.analytics.client.api.browser.BrowserEvent;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugins.rest.common.security.AnonymousAllowed;
import org.codehaus.jackson.annotate.JsonIgnoreProperties;
import org.codehaus.jackson.annotate.JsonProperty;

import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.util.List;
import java.util.Map;

/**
 * Resource for publishing events into the server-side event system in Confluence.
 */
@Path("/publish")
@AnonymousAllowed
public class PublisherResource
{
    private final EventPublisher eventPublisher;
    private final TimeKeeper timeKeeper;

    public PublisherResource(EventPublisher eventPublisher, TimeKeeper timeKeeper)
    {
        this.eventPublisher = eventPublisher;
        this.timeKeeper = timeKeeper;
    }

    /**
     * Publishes a single {@link BrowserEvent}.
     *
     * @param bean the event
     * @return success
     */
    @POST
    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    public Response publishSingleEvent(final BrowserEventBean bean)
    {
        final BrowserEvent event = toBrowserEvent(bean);
        eventPublisher.publish(event);
        return Response.ok().build();
    }

    /**
     * Publishes a list of {@link BrowserEvent}s.
     *
     * @param beans the events
     * @return success
     */
    @Path("/bulk")
    @POST
    @Consumes({MediaType.APPLICATION_JSON})
    @Produces({MediaType.APPLICATION_JSON})
    public Response publishBulkEvents(final List<BrowserEventBean> beans)
    {
        for (BrowserEventBean bean : beans)
        {
            final BrowserEvent event = toBrowserEvent(bean);
            eventPublisher.publish(event);
        }
        return Response.ok().build();
    }

    private BrowserEvent toBrowserEvent(final BrowserEventBean bean)
    {
        return new BrowserEvent(bean.name, bean.properties, timeKeeper.currentTimeMillis() + bean.timeDelta);
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    static class BrowserEventBean
    {
        @JsonProperty
        String name;

        @JsonProperty
        Map<String, Object> properties;

        @JsonProperty
        long timeDelta;

        BrowserEventBean() {}

        BrowserEventBean(final String name, final Map<String, Object> properties, final long timeDelta)
        {
            this.name = name;
            this.properties = properties;
            this.timeDelta = timeDelta;
        }
    }
}