package com.atlassian.plugins.codegen.annotations.asm;

import org.apache.commons.io.FileUtils;
import org.objectweb.asm.ClassReader;
import org.objectweb.asm.ClassVisitor;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * @since 3.6
 */
public abstract class AbstractAnnotationParser {

    public void parse(String basePackage, ClassVisitor classVisitor) throws Exception {
        ClassLoader classLoader = Thread.currentThread()
                .getContextClassLoader();

        String path = basePackage.replace('.', '/');

        Enumeration<URL> resources = classLoader.getResources(path);
        while (resources.hasMoreElements()) {
            URL resource = resources.nextElement();
            processResource(resource, basePackage, classVisitor);
        }
    }

    protected void processResource(URL resource, String packageName, ClassVisitor classVisitor) throws IOException {
        if (resource.getProtocol()
                .equals("file")) {
            processFileDirectory(new File(resource.getFile()), packageName, classVisitor);
        } else if (resource.getProtocol()
                .equals("jar")) {
            JarURLConnection conn = (JarURLConnection) resource.openConnection();
            processJarDirectory(conn.getJarFile(), packageName, classVisitor);
        }
    }

    protected void processJarDirectory(JarFile jarFile, String packageName, ClassVisitor classVisitor) throws IOException {
        Enumeration<JarEntry> entries = jarFile.entries();
        String basePath = packageName.replace('.', '/');
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            String name = entry.getName();
            if (name.startsWith(basePath) && !entry.isDirectory() && name.endsWith(".class")) {
                try (InputStream is = jarFile.getInputStream(entry)) {
                    processClassFile(is, classVisitor);
                }
            }
        }
    }

    protected void processFileDirectory(File directory, String packageName, ClassVisitor classVisitor) throws IOException {
        if (!directory.exists()) {
            return;
        }
        File[] files = directory.listFiles();
        for (File file : files) {
            if (file.isDirectory()) {
                assert !file.getName()
                        .contains(".");
                processFileDirectory(file, packageName + "." + file.getName(), classVisitor);
            } else if (file.getName()
                    .endsWith(".class")) {
                try (InputStream is = FileUtils.openInputStream(file)) {
                    processClassFile(is, classVisitor);
                }

            }
        }
    }

    protected void processClassFile(InputStream is, ClassVisitor classVisitor) throws IOException {
        ClassReader classReader = new ClassReader(is);
        classReader.accept(classVisitor, ClassReader.SKIP_CODE | ClassReader.SKIP_DEBUG | ClassReader.SKIP_FRAMES);
    }

    protected static String normalize(String name) {
        if (name == null) {
            return null;
        }

        if (name.startsWith("L") && name.endsWith(";")) {
            name = name.substring(1, name.length() - 1);
        }

        if (name.endsWith(".class")) {
            name = name.substring(0, name.length() - ".class".length());
        }

        return name.replace('/', '.');
    }
}
