/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.maven.plugins.amps.util;

import com.atlassian.maven.plugins.amps.util.FileUtils;
import com.atlassian.maven.plugins.amps.util.StreamUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Enumeration;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ZipUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(ZipUtils.class);
    private static final int MASK_EXECUTABLE = 64;
    private static final int MODE_EXECUTABLE = 448;

    public static void untargz(File targz, String destDir, int leadingPathSegmentsToTrim) throws IOException {
        try (FileInputStream fin = new FileInputStream(targz);
             GzipCompressorInputStream gzIn = new GzipCompressorInputStream((InputStream)fin);
             TarArchiveInputStream tarIn = new TarArchiveInputStream((InputStream)gzIn);){
            TarArchiveEntry entry = tarIn.getNextTarEntry();
            while (entry != null) {
                File entryFile = new File(destDir + File.separator + ZipUtils.trimPathSegments(entry.getName(), leadingPathSegmentsToTrim));
                if (entry.isDirectory()) {
                    ZipUtils.makeDirectories(entryFile);
                } else {
                    ZipUtils.copyFile(tarIn, entry, entryFile);
                }
                entry = tarIn.getNextTarEntry();
            }
        }
    }

    private static void copyFile(TarArchiveInputStream tarIn, TarArchiveEntry entry, File entryFile) throws IOException {
        if (!entryFile.getParentFile().exists()) {
            ZipUtils.makeDirectories(entryFile.getParentFile());
        }
        try (FileOutputStream fos = new FileOutputStream(entryFile);){
            IOUtils.copy((InputStream)tarIn, (OutputStream)fos);
            if (ZipUtils.isExecutable(entry)) {
                FileUtils.makeExecutable(entryFile);
            }
        }
    }

    private static void makeDirectories(File file) {
        if (!file.mkdirs()) {
            LOGGER.debug("Did not create directories for {}", (Object)file);
        }
    }

    public static void unzip(File zipFile, String destDir) throws IOException {
        ZipUtils.unzip(zipFile, destDir, 0);
    }

    public static void unzip(File zipFile, String destDir, int leadingPathSegmentsToTrim) throws IOException {
        ZipUtils.unzip(zipFile, destDir, leadingPathSegmentsToTrim, false, null);
    }

    public static void unzip(File zipFile, String destDir, int leadingPathSegmentsToTrim, boolean flatten, Pattern pattern) throws IOException {
        try (ZipFile zip = ((ZipFile.Builder)ZipFile.builder().setFile(zipFile)).get();){
            Enumeration entries = zip.getEntries();
            while (entries.hasMoreElements()) {
                ZipArchiveEntry zipEntry = (ZipArchiveEntry)entries.nextElement();
                if (!ZipUtils.matches(zipEntry, pattern)) continue;
                File file = ZipUtils.getFile(destDir, leadingPathSegmentsToTrim, flatten, (ZipEntry)zipEntry);
                if (zipEntry.isDirectory()) {
                    ZipUtils.makeDirectories(file);
                    continue;
                }
                ZipUtils.copyFile(zip, zipEntry, file);
            }
        }
    }

    private static boolean matches(ZipArchiveEntry zipEntry, Pattern pattern) {
        return pattern == null || pattern.matcher(zipEntry.getName()).matches();
    }

    private static void copyFile(ZipFile zip, ZipArchiveEntry zipEntry, File file) throws IOException {
        if (!file.getParentFile().exists()) {
            ZipUtils.makeDirectories(file.getParentFile());
        }
        try (InputStream is = zip.getInputStream(zipEntry);
             OutputStream fos = Files.newOutputStream(file.toPath(), new OpenOption[0]);){
            IOUtils.copy((InputStream)is, (OutputStream)fos);
            if (ZipUtils.isExecutable(zipEntry)) {
                FileUtils.makeExecutable(file);
            }
        }
        FileUtils.setLastModified(file, zipEntry.getTime());
    }

    private static File getFile(String destDir, int leadingPathSegmentsToTrim, boolean flatten, ZipEntry zipEntry) {
        String zipPath = ZipUtils.trimPathSegments(zipEntry.getName(), leadingPathSegmentsToTrim);
        if (flatten) {
            zipPath = ZipUtils.flattenPath(zipPath);
        }
        return new File(destDir + File.separatorChar + zipPath);
    }

    private static boolean isExecutable(TarArchiveEntry tarArchiveEntry) {
        return ZipUtils.isExecutable(tarArchiveEntry.getMode());
    }

    private static boolean isExecutable(ZipArchiveEntry zipArchiveEntry) {
        return ZipUtils.isExecutable(zipArchiveEntry.getUnixMode());
    }

    private static boolean isExecutable(int mode) {
        return (mode & 0x40) != 0;
    }

    public static int countNestingLevel(File zip) throws IOException {
        try (ZipFile zipFile = ((ZipFile.Builder)ZipFile.builder().setFile(zip)).get();){
            List<String> filenames = ZipUtils.toList(zipFile.getEntries());
            int n = ZipUtils.countNestingLevel(filenames);
            return n;
        }
    }

    static int countNestingLevel(List<String> filenames) {
        String prefix = StringUtils.getCommonPrefix((String[])filenames.toArray(new String[0]));
        if (!prefix.endsWith("/")) {
            prefix = prefix.substring(0, prefix.lastIndexOf(47) + 1);
        }
        if (filenames.remove(prefix)) {
            return ZipUtils.countNestingLevel(filenames);
        }
        filenames.clear();
        return StringUtils.countMatches((CharSequence)prefix, (CharSequence)"/");
    }

    private static List<String> toList(Enumeration<? extends ZipEntry> entries) {
        return StreamUtils.stream(entries).map(ZipEntry::getName).collect(Collectors.toList());
    }

    public static void zipChildren(File zipFile, File rootDir) throws IOException {
        Path root = rootDir.toPath();
        try (ZipArchiveOutputStream out = new ZipArchiveOutputStream(zipFile);
             Stream<Path> children = Files.walk(root, new FileVisitOption[0]).skip(1L);){
            byte[] buffer = new byte[8192];
            for (Path child : children::iterator) {
                BasicFileAttributes attributes = Files.readAttributes(child, BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS);
                if (attributes.isOther() || attributes.isSymbolicLink()) continue;
                String path = root.relativize(child).toString();
                if (SystemUtils.IS_OS_WINDOWS) {
                    path = path.replace('\\', '/');
                }
                if (attributes.isDirectory()) {
                    path = path + "/";
                }
                ZipArchiveEntry entry = new ZipArchiveEntry(path);
                out.putArchiveEntry(entry);
                if (attributes.isRegularFile()) {
                    if (Files.isExecutable(child)) {
                        entry.setUnixMode(448);
                    }
                    try (InputStream content = Files.newInputStream(child, new OpenOption[0]);){
                        ZipUtils.copyViaBuffer(content, (OutputStream)out, buffer);
                    }
                }
                out.closeArchiveEntry();
            }
        }
    }

    public static void zipDir(File zipFile, File srcDir, String prefix) throws IOException {
        try (ZipArchiveOutputStream out = new ZipArchiveOutputStream(zipFile);){
            ZipUtils.addZipPrefixes(srcDir, out, prefix);
            ZipUtils.addZipDir(srcDir, out, prefix);
        }
    }

    private static void addZipPrefixes(File dirObj, ZipArchiveOutputStream out, String prefix) throws IOException {
        String entryPrefix = ZipUtils.ensurePrefixWithSlash(dirObj, prefix);
        String[] prefixes = entryPrefix.split("/");
        String lastPrefix = "";
        for (String p : prefixes) {
            ZipArchiveEntry entry = new ZipArchiveEntry(lastPrefix + p + "/");
            out.putArchiveEntry(entry);
            out.closeArchiveEntry();
            lastPrefix = p + "/";
        }
    }

    private static void addZipDir(File dirObj, ZipArchiveOutputStream out, String prefix) throws IOException {
        File[] files = dirObj.listFiles();
        if (files == null || files.length == 0) {
            return;
        }
        byte[] tmpBuf = new byte[8192];
        String entryPrefix = ZipUtils.ensurePrefixWithSlash(dirObj, prefix);
        for (File currentFile : files) {
            String entryName;
            if (currentFile.isDirectory()) {
                entryName = entryPrefix + currentFile.getName() + "/";
                ZipArchiveEntry entry = new ZipArchiveEntry(entryName);
                out.putArchiveEntry(entry);
                out.closeArchiveEntry();
                ZipUtils.addZipDir(currentFile, out, entryName);
                continue;
            }
            if (!currentFile.isFile()) continue;
            entryName = entryPrefix + currentFile.getName();
            try (FileInputStream in = new FileInputStream(currentFile.getAbsolutePath());){
                ZipArchiveEntry entry = new ZipArchiveEntry(entryName);
                out.putArchiveEntry(entry);
                if (currentFile.canExecute()) {
                    entry.setUnixMode(448);
                }
                ZipUtils.copyViaBuffer(in, (OutputStream)out, tmpBuf);
                out.closeArchiveEntry();
            }
        }
    }

    private static void copyViaBuffer(InputStream inputStream, OutputStream outputStream, byte[] buffer) throws IOException {
        int read;
        while ((read = inputStream.read(buffer)) != -1) {
            outputStream.write(buffer, 0, read);
        }
    }

    private static String ensurePrefixWithSlash(File rootDir, String prefix) {
        String entryPrefix = prefix;
        if (StringUtils.isNotBlank((CharSequence)entryPrefix) && !entryPrefix.equals("/")) {
            if (entryPrefix.charAt(0) == '/') {
                entryPrefix = entryPrefix.substring(1);
            }
            if (entryPrefix.charAt(entryPrefix.length() - 1) != '/') {
                entryPrefix = entryPrefix + "/";
            }
        } else {
            entryPrefix = rootDir.getName() + "/";
        }
        return entryPrefix;
    }

    private static String trimPathSegments(String zipPath, int trimLeadingPathSegments) {
        int nextSlash;
        int startIndex = 0;
        for (int i = 0; i < trimLeadingPathSegments && (nextSlash = zipPath.indexOf(47, startIndex)) != -1; ++i) {
            startIndex = nextSlash + 1;
        }
        return zipPath.substring(startIndex);
    }

    private static String flattenPath(String zipPath) {
        return zipPath.substring(Math.max(zipPath.lastIndexOf(47), 0));
    }

    private ZipUtils() {
    }
}

