package com.atlassian.adf.util;

import com.atlassian.adf.model.Element;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import javax.annotation.Nullable;
import java.net.URI;
import java.net.URL;
import java.util.Map;
import java.util.Optional;

import static com.atlassian.adf.model.Element.nonNull;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.UrlUtil.validateUrl;

/**
 * Utility/support methods common to parsing nodes or marks.
 */
@Internal
@ReturnValuesAreNonnullByDefault
public abstract class ParserSupport {

    private ParserSupport() {
        // static-only class
    }

    /**
     * Extract a value from the given map, assuming that it will be of the expected type (or {@code null}.
     */
    @Nullable
    public static <T> T get(Map<String, ?> map, String key) {
        return unsafeCast(map.get(key));
    }

    /**
     * Extract a value from the given map, requiring that it be of the given type (or {@code null}.
     */
    public static <T> T getOrThrow(Map<String, ?> map, String key) {
        return unsafeCast(nonNull(map.get(key), key));
    }

    /**
     * Verify that the {@link Element.Key#TYPE type} value of the given map is correct.
     * <p>
     * Since most of the parser methods are resolved from a lookup map that keys off of the type in
     * the first place, this would indicate a serious internal error in the parser code, which is
     * why it is not handled as an {@code AdfException}.
     *
     * @throws IllegalArgumentException if the {@code type} value is not the {@code expectedType}
     */
    public static void checkType(Map<String, ?> map, String expectedType) {
        String type = getTypeOrThrow(map);
        if (!expectedType.equals(type)) {
            throw new IllegalArgumentException("The parser for type '" + expectedType +
                    "' was provided a map with type '" + type + '\'');
        }
    }

    /**
     * Extract the {@link Element.Key#TYPE type} value from the given map.
     *
     * @throws AdfException.MissingType if the map does not have a {@code type} field, it isn't a string, or it is empty.
     */
    public static String getTypeOrThrow(Map<String, ?> map) {
        Object type = map.get(Element.Key.TYPE);
        if (type instanceof String) {
            String s = (String) type;
            if (s.isEmpty()) {
                throw new AdfException.MissingType();
            }
            return s;
        }
        throw new AdfException.MissingType();
    }

    public static <T> Optional<T> getAttr(Map<String, ?> map, String attr) {
        Map<String, ?> attrs = unsafeCast(map.get(Element.Key.ATTRS));
        if (attrs == null) {
            return Optional.empty();
        }

        T value = unsafeCast(attrs.get(attr));
        return Optional.ofNullable(value);
    }

    public static <T> Optional<T> getAttr(Map<String, ?> map, String attr, Class<T> requiredClass) {
        return getAttr(map, attr).map(requiredClass::cast);
    }

    public static Optional<Integer> getAttrInt(Map<String, ?> map, String attr) {
        return getAttr(map, attr)
                .map(Number.class::cast)
                .map(value -> asInt(value, attr));
    }

    public static Optional<Double> getAttrDouble(Map<String, ?> map, String attr) {
        return getAttr(map, attr)
                .map(Number.class::cast)
                .map(Number::doubleValue);
    }

    public static Optional<Number> getAttrNumber(Map<String, ?> map, String attr) {
        return getAttr(map, attr, Number.class)
                .map(ParserSupport::optimizeNumber);
    }

    public static String getAttrNonEmpty(Map<String, ?> map, String attr) {
        String s = getAttrOrThrow(map, attr);
        if (s.isEmpty()) {
            throw new AdfException.EmptyProperty(attr);
        }
        return s;
    }

    /**
     * Parses the given value as an integer, tolerating lossless conversions from other number formats
     * when possible.
     *
     * @param value   the numeric value to be mapped to an {@code int} if possible
     * @param attrKey the attribute key from which the value was read
     * @return the resulting {@code int} value
     * @throws AdfException.ValueTypeMismatch if the original value cannot be represented as an {@code int},
     *                                        such as a {@code Long} that is {@code >= Integer.MAX_VALUE}.
     */
    public static int asInt(Number value, String attrKey) {
        int x = value.intValue();
        if (value instanceof Integer || value instanceof Short || value instanceof Byte) {
            return x;
        } else if (value instanceof Long) {
            if ((long) x == value.longValue()) {
                return x;
            }
        } else if (value instanceof Double || value instanceof Float) {
            double floor = Math.floor(value.doubleValue());
            if ((double) x == floor) {
                return x;
            }
        }
        throw new AdfException.ValueTypeMismatch(attrKey, "int", value.getClass().getSimpleName());
    }

    public static int getAttrIntOrThrow(Map<String, ?> map, String attr) {
        return asInt(getAttrOrThrow(map, attr), attr);
    }

    public static Number getAttrNumberOrThrow(Map<String, ?> map, String attr) {
        return optimizeNumber(getAttrOrThrow(map, attr, Number.class));
    }

    public static <T> T getAttrOrThrow(Map<String, ?> map, String attr) {
        Optional<T> value = getAttr(map, attr);
        return value.orElseThrow(() -> new AdfException.MissingProperty(attr));
    }

    public static <T> T getAttrOrThrow(Map<String, ?> map, String attr, Class<T> requiredClass) {
        Optional<T> value = getAttr(map, attr, requiredClass);
        return value.orElseThrow(() -> new AdfException.MissingProperty(attr));
    }

    private static Number optimizeNumber(Number num) {
        if (num instanceof Double) {
            long longValue = num.longValue();
            if (num.doubleValue() != (double) longValue) {
                return num;
            }
            num = longValue;
        }

        if (num instanceof Long) {
            int intValue = num.intValue();
            if (num.longValue() == (long) intValue) {
                return intValue;
            }
        }

        return num;
    }

    public static String cleanUri(String uri, String attrKey) {
        return validateUrl(uri, attrKey);
    }

    public static String cleanUri(URL url, String attrKey) {
        return validateUrl(url.toString(), attrKey);
    }

    public static String cleanUri(URI uri, String attrKey) {
        return validateUrl(uri.toString(), attrKey);
    }
}
