package com.atlassian.adf.util;

import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.adf.util.Cast.unsafeCast;
import static java.util.Objects.requireNonNull;

/**
 * A mutable {@code Map<String, Object>} with a few helpful additions for constructing the
 * representation of ADF nodes and marks.
 */
@Internal
@ReturnValuesAreNonnullByDefault
public class FieldMap extends LinkedHashMap<String, Object> {
    private static final long serialVersionUID = 1L;

    public static FieldMap map() {
        return new FieldMap();
    }

    public static FieldMap map(Map<String, ?> source) {
        return new FieldMap().addAll(source);
    }

    public static FieldMap map(
            String k1, Object v1
    ) {
        requireNonNull(k1, "k1");
        requireNonNull(v1, "v1");
        return map().add(k1, v1);
    }

    public static FieldMap map(
            String k1, Object v1,
            String k2, Object v2
    ) {
        FieldMap map = map(k1, v1);
        requireNonNull(k2, "k2");
        requireNonNull(v2, "v2");
        return map.add(k2, v2);
    }

    public static FieldMap map(
            String k1, Object v1,
            String k2, Object v2,
            String k3, Object v3
    ) {
        FieldMap map = map(k1, v1, k2, v2);
        requireNonNull(k3, "k3");
        requireNonNull(v3, "v3");
        return map.add(k3, v3);
    }

    public static FieldMap map(
            String k1, Object v1,
            String k2, Object v2,
            String k3, Object v3,
            String k4, Object v4
    ) {
        FieldMap map = map(k1, v1, k2, v2, k3, v3);
        requireNonNull(k4, "k4");
        requireNonNull(v4, "v4");
        return map.add(k4, v4);
    }

    public FieldMap add(String k, Object v) {
        requireNonNull(k, "k");
        requireNonNull(v, "v");
        putUniqueOrThrow(k, v);
        return this;
    }

    public <T> FieldMap addMapped(String k, T v, Function<T, ?> mapper) {
        requireNonNull(v, "v");
        Optional.of(v)
                .map(mapper)
                .ifPresent(value -> putUniqueOrThrow(k, value));
        return this;
    }

    public FieldMap addAll(Map<String, ?> source) {
        source.forEach(this::add);
        return this;
    }

    public FieldMap addIfPresent(String k, @Nullable Object v) {
        if (v != null) {
            putUniqueOrThrow(k, v);
        }
        return this;
    }

    public <T> FieldMap addMappedIfPresent(String k, @Nullable T v, Function<T, ?> mapper) {
        Optional.ofNullable(v)
                .map(mapper)
                .ifPresent(value -> putUniqueOrThrow(k, value));
        return this;
    }

    public FieldMap addIf(boolean guard, String k, Supplier<?> vSupplier) {
        return guard ? addIfPresent(k, vSupplier.get()) : this;
    }

    private void putUniqueOrThrow(String field, Object value) {
        if (putIfAbsent(field, value) != null) {
            throw new AdfException.DuplicateProperty(field);
        }
    }

    public FieldMap let(Consumer<FieldMap> effect) {
        effect.accept(this);
        return this;
    }

    public FieldMap deepCopy() {
        return deepCopy(this);
    }

    public static FieldMap deepCopy(Map<String, ?> value) {
        DeepCopy deepCopy = new DeepCopy(false);
        FieldMap copy = map();
        value.forEach((k, v) -> copy.put(k, deepCopy.copyObj(v)));
        return copy;
    }

    public Map<String, ?> immutableCopy() {
        return immutableCopy(this);
    }

    public static Map<String, ?> immutableCopy(Map<String, ?> map) {
        DeepCopy deepCopy = new DeepCopy(true);
        FieldMap copy = map();
        map.forEach((k, v) -> copy.put(k, deepCopy.copyObj(v)));
        return Collections.unmodifiableMap(copy);
    }


    private static class DeepCopy {
        private final boolean immutable;

        private DeepCopy(boolean immutable) {
            this.immutable = immutable;
        }

        Object copyObj(Object value) {
            if (value instanceof Map<?, ?>) {
                Map<String, ?> orig = unsafeCast(value);
                FieldMap copy = map();
                orig.forEach((k, v) -> copy.put(k, copyObj(v)));
                // Don't use Map.of because we want to preserve the order
                return immutable ? Collections.unmodifiableMap(copy) : copy;
            }

            if (value instanceof Collection<?>) {
                List<?> orig = unsafeCast(value);
                List<Object> copy = new ArrayList<>(orig.size());
                orig.forEach(v -> copy.add(copyObj(v)));
                return immutable ? List.copyOf(copy) : copy;
            }

            if (value instanceof String ||
                    value instanceof Number ||
                    value instanceof Boolean) {
                return value;
            }

            throw new IllegalArgumentException("Cannot deep copy: " + value.getClass().getName());
        }
    }
}