package com.atlassian.adf.util;

import com.atlassian.adf.model.Element;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;

/**
 * Supporting class for the parser that resolves the {@code type} of an element to its corresponding
 * node or mark class as well as a sub-parser that can reconstruct the element from its map value.
 *
 * @param <T> the type of element returned by this factory (either a node or a mark)
 */
@Internal
@ReturnValuesAreNonnullByDefault
public class Factory<T extends Element> {
    private final String type;
    private final Class<T> typeClass;
    private final Function<Map<String, ?>, ? extends T> parser;

    public Factory(
            String type,
            Class<T> typeClass,
            Function<Map<String, ?>, ? extends T> parser
    ) {
        this.type = requireNonNull(type, "type");
        this.typeClass = requireNonNull(typeClass, "typeClass");
        this.parser = requireNonNull(parser, "parser");
    }

    public static <T extends Element, U extends T> Factory<T> unsupported(
            Class<T> typeClass,
            Function<Map<String, ?>, U> parser) {
        return new Factory<>("", typeClass, parser);
    }

    /**
     * @return the value used to identify elements of this type, or an empty string if this factory creates
     * opaque representations for an unsupported element type
     */
    public String type() {
        return type;
    }

    /**
     * @return the class of elements that this factory returns
     */
    public Class<T> typeClass() {
        return typeClass;
    }

    /**
     * Parses the given value map, assuming that it is a well-formed representation of an element of the type
     * that this factory can produce.
     *
     * @param map the value map to be parsed
     * @return the reconstructed element
     */
    public T parse(Map<String, ?> map) {
        return parser.apply(map);
    }

    /**
     * Given a group of factories, builds a lookup map that can resolve each factory's {@link #typeClass()} to the
     * corresponding factory.
     *
     * @param keyExtractor the function for extracting the lookup key for the factory lookup map
     * @param factories    all factories to include in the lookup map
     * @param <K>          the inferred key type for the resulting lookup map
     * @param <T>          the inferred element type returned by this factory
     * @return the (immutable) lookup map
     */
    @SafeVarargs
    public static <K, T extends Element> Map<K, Factory<? extends T>> extractLookupMap(
            Function<Factory<? extends T>, K> keyExtractor,
            Factory<? extends T>... factories
    ) {
        requireNonNull(keyExtractor, "keyExtractor");
        Map<K, Factory<? extends T>> map = new HashMap<>();
        for (Factory<? extends T> factory : factories) {
            K key = requireNonNull(keyExtractor.apply(factory), "key");
            if (map.put(key, factory) != null) {
                throw new IllegalStateException("Duplicate key: " + key);
            }
        }
        return Map.copyOf(map);
    }
}
