package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.TaskListException;
import com.atlassian.adf.model.node.TaskItem.State;
import com.atlassian.adf.model.node.type.*;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Task lists provide lists that look much like a {@link BulletList bulletList}, but there are differences in
 * their visual representation. They can be nested (although at least one {@link TaskItem taskItem} has to
 * be provided before any nested task lists). The task items also require identifier strings that can be used
 * to associate them with information in other systems.
 * <p>
 * Note: Jira does not currently support task lists.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 *     {@link TaskList#taskList(String, TaskListContent...) taskList}(
 *         "test-list-id",
 *         {@link TaskItem#taskItem() taskItem}()
 *                 .{@link TaskItem.Partial.NeedsLocalIdAndState#todo() todo}()
 *                 .{@link TaskItem.Partial.NeedsLocalId#localId(String) localId}("test-id1")
 *                 .{@link TaskItem#content(String) content}("Do this!"),
 *         {@link TaskItem#taskItem() taskItem}()
 *                 .{@link TaskItem.Partial.NeedsLocalIdAndState#localId(String) localId}("test-id2")
 *                 .{@link TaskItem.Partial.NeedsState#state(State) state}({@link State#DONE DONE})
 *                 .{@link TaskItem#content(String) content}("This is completed")
 *     );
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 * {
 *   "type": "taskList",
 *   "attrs": {
 *     "localId": "test-list-id"
 *   },
 *   "content": [
 *     {
 *       "type": "taskItem",
 *       "attrs": {
 *         "localId": "test-id1",
 *         "state": "TODO"
 *       },
 *       "content": [
 *         {
 *           "type": "text",
 *           "text": "Do this!"
 *         }
 *       ]
 *     },
 *     {
 *       "type": "taskItem",
 *       "attrs": {
 *         "localId": "test-id2",
 *         "state": "DONE"
 *       },
 *       "content": [
 *         {
 *           "type": "text",
 *           "text": "This is completed"
 *         }
 *       ]
 *     }
 *   ]
 * }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <div data-task-local-id="test-id1">
 *     <div><input name="test-id1-0" type="checkbox"> Do this!</div>
 *     <div><input name="test-id2-0" type="checkbox" checked> This is completed</div>
 * </div>
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the task list, so while it gives a vague
 * impression of what a "task list" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products. In particular, the checkboxes are displayed here without the proper styling.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class TaskList
        extends AbstractContentNode<TaskList, TaskListContent>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent, TaskListContent {

    static final Factory<TaskList> FACTORY = new Factory<>(
            Type.TASK_LIST,
            TaskList.class,
            TaskList::parse
    );

    private String localId;

    private TaskList(String localId) {
        this.localId = requireNonNull(localId, "localId");
    }

    public static TaskList taskList(String localId) {
        return new TaskList(localId);
    }

    public static TaskList taskList(String localId, TaskListContent content) {
        return taskList(localId).content(content);
    }

    public static TaskList taskList(String localId, TaskListContent... content) {
        return taskList(localId).content(content);
    }

    public static TaskList taskList(String localId, Iterable<? extends TaskListContent> content) {
        return taskList(localId).content(content);
    }

    public static TaskList taskList(String localId, Stream<? extends TaskListContent> content) {
        return taskList(localId).content(content);
    }

    public TaskList localId(String localId) {
        this.localId = requireNonNull(localId, "localId");
        return this;
    }

    public String localId() {
        return localId;
    }

    @Override
    public TaskList copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TASK_LIST;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .add(Key.ATTRS, map(Attr.LOCAL_ID, localId));
    }

    @Override
    protected boolean contentNodeEquals(TaskList other) {
        return localId.equals(other.localId);
    }

    @Override
    protected int contentNodeHashCode() {
        return localId.hashCode();
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("localId", localId);
    }

    private void validateFirstContentItemIsNotAList() {
        requireNotEmpty();
        TaskListContent firstContentItem = content.get(0);
        if (firstContentItem instanceof TaskList) {
            throw new TaskListException.CannotUseNestedTaskListAsFirstContentItem();
        }
    }

    @Override
    protected void validateContentNodeForAppend(TaskListContent node) {
        if (content.isEmpty() && node instanceof TaskList) {
            throw new TaskListException.CannotUseNestedTaskListAsFirstContentItem();
        }
        super.validateContentNodeForAppend(node);
    }

    @Override
    protected void contentNodeValidate() {
        validateFirstContentItemIsNotAList();
    }

    private static TaskList parse(Map<String, ?> map) {
        checkType(map, Type.TASK_LIST);
        return taskList(getAttrOrThrow(map, Attr.LOCAL_ID))
                .parseRequiredContent(map, TaskListContent.class);
    }
}
