package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.model.node.type.TaskListContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.Objects;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Provides the content for one of the items in a {@link TaskList taskList}.
 * <h2>Example</h2>
 * See {@link TaskList taskList}
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class TaskItem
        extends AbstractContentNode<TaskItem, InlineContent>
        implements TaskListContent {

    private static final String PREFIX_DONE = "[x] ";
    private static final String PREFIX_TODO = "[] ";

    static final Factory<TaskItem> FACTORY = new Factory<>(Type.TASK_ITEM, TaskItem.class, TaskItem::parse);

    private String localId;
    private State state;

    private TaskItem(String localId, State state) {
        this.localId = requireNonNull(localId, "localId");
        this.state = requireNonNull(state, "state");
    }

    public static Partial.NeedsLocalIdAndState taskItem() {
        return new Partial.NeedsLocalIdAndState();
    }

    public static TaskItem todo(String localId) {
        return new TaskItem(localId, State.TODO);
    }

    public static TaskItem done(String localId) {
        return new TaskItem(localId, State.DONE);
    }

    public TaskItem content(String content) {
        return content(text(content));
    }

    public TaskItem content(String... content) {
        return content(text(content));
    }

    public TaskItem localId(String localId) {
        this.localId = requireNonNull(localId, "localId");
        return this;
    }

    public String localId() {
        return localId;
    }

    public TaskItem state(State state) {
        this.state = requireNonNull(state, "state");
        return this;
    }

    public State state() {
        return state;
    }

    public TaskItem todo() {
        return state(State.TODO);
    }

    public TaskItem done() {
        return state(State.DONE);
    }

    @Override
    public TaskItem copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TASK_ITEM;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .let(this::addContentIfPresent)
                .add(Key.ATTRS, map()
                        .add(Attr.LOCAL_ID, localId)
                        .add(Attr.STATE, state.name())
                );
    }

    @Override
    protected boolean contentNodeEquals(TaskItem other) {
        return localId.equals(other.localId)
                && state.equals(other.state);
    }

    @Override
    protected int contentNodeHashCode() {
        return Objects.hash(localId, state);
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("localId", localId);
        buf.appendField("state", state);
    }

    public static TaskItem parse(Map<String, ?> map) {
        String localId = getAttrOrThrow(map, Attr.LOCAL_ID);
        State state = State.PARSER.parse(getAttrOrThrow(map, Attr.STATE));
        return new TaskItem(localId, state)
                .parseOptionalContent(map, InlineContent.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String prefix = (this.state == TaskItem.State.DONE)
                ? PREFIX_DONE
                : PREFIX_TODO;
        sb.append(prefix);
        appendPlainTextInlineContent(sb);
    }

    /**
     * Types that represent a partially constructed {@link TaskItem taskItem}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code taskItem} still needs its {@code state} attribute set.
         */
        class NeedsState {
            private final String localId;

            NeedsState(String localId) {
                this.localId = requireNonNull(localId, "localId");
            }

            public TaskItem state(String state) {
                requireNonNull(state, "state");
                return state(State.PARSER.parse(state));
            }

            public TaskItem state(State state) {
                requireNonNull(state, "state");
                return new TaskItem(localId, state);
            }

            public TaskItem todo() {
                return state(State.TODO);
            }

            public TaskItem done() {
                return state(State.DONE);
            }
        }

        /**
         * This partially constructed {@code taskItem} still needs its {@code localId} attribute set.
         */
        class NeedsLocalId {
            private final State state;

            NeedsLocalId(State state) {
                this.state = requireNonNull(state, "state");
            }

            public TaskItem localId(String localId) {
                return new TaskItem(localId, state);
            }
        }

        /**
         * This partially constructed {@code taskItem} still needs its {@code localId} and {@code state} attributes
         * set (and this builder permits them to be set in either order).
         */
        class NeedsLocalIdAndState {
            NeedsLocalIdAndState() {
            }

            public NeedsState localId(String localId) {
                return new NeedsState(localId);
            }

            public NeedsLocalId state(String state) {
                return state(State.PARSER.parse(state));
            }

            public NeedsLocalId state(State state) {
                return new NeedsLocalId(state);
            }

            public NeedsLocalId todo() {
                return state(State.TODO);
            }

            public NeedsLocalId done() {
                return state(State.DONE);
            }
        }
    }

    public enum State {
        TODO, DONE;

        static final EnumParser<State> PARSER = new EnumParser<>(State.class, State::name);
    }
}
