package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.model.node.type.TableRowContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * Defines rows within a table and is a container for {@link TableHeader table heading} and
 * {@link TableCell table cell} nodes.
 * <div style="color: black; background-color: #fffae6; border-radius: 3px; padding: 16px; width: 75%;">
 * \u26A0\uFE0F <strong>WARNING</strong>: Tables are documented as supported on <strong>web</strong> and
 * <strong>desktop</strong> only. <strong>Mobile</strong> rendering support for tables may be restricted
 * or entirely unavailable.</div>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #tr(TableRowContent[]) tr}(
 *         {@link TableHeader#th(TableCellContent[]) th}(
 *                 {@link Paragraph#p(InlineContent) p}(
 *                         {@link Text#text(String) text}("Heading one").{@link Text#strong() strong}()
 *                 )
 *         )
 * )
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "tableRow",
 *     "content": [
 *       {
 *         "type": "tableHeader",
 *         "attrs": {},
 *         "content": [
 *           {
 *             "type": "paragraph",
 *             "content": [
 *               {
 *                 "type": "text",
 *                 "text": "Heading one",
 *                 "marks": [
 *                   {
 *                     "type": "strong"
 *                   }
 *                 ]
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <table summary="">
 *     <tr><th><p><strong>Heading one</strong></p></th></tr>
 * </table>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/tableRow/">Node - tableRow</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class TableRow extends AbstractContentNode<TableRow, TableRowContent> {

    static Factory<TableRow> FACTORY = new Factory<>(Type.TABLE_ROW, TableRow.class, TableRow::parse);

    private TableRow() {
    }

    /**
     * @return a new, empty table row. At least one {@link TableHeader} or {@link TableCell} must be added as
     * content to make the table row valid.
     */
    public static TableRow tr() {
        return new TableRow();
    }

    /**
     * @return a new table row with the given content
     */
    public static TableRow tr(TableRowContent content) {
        return tr().content(content);
    }

    /**
     * @return a new table row with the given content
     */
    public static TableRow tr(TableRowContent... content) {
        return tr().content(content);
    }

    /**
     * @return a new table row with the given content
     */
    public static TableRow tr(Iterable<? extends TableRowContent> content) {
        return tr().content(content);
    }

    /**
     * @return a new table row with the given content
     */
    public static TableRow tr(Stream<? extends TableRowContent> content) {
        return tr().content(content);
    }

    /**
     * @see #tr()
     */
    public static TableRow tableRow() {
        return new TableRow();
    }

    /**
     * @see #tr(TableRowContent)
     */
    public static TableRow tableRow(TableRowContent content) {
        return tr().content(content);
    }

    /**
     * @see #tr(TableRowContent[])
     */
    public static TableRow tableRow(TableRowContent... content) {
        return tr().content(content);
    }

    /**
     * @see #tr(Iterable)
     */
    public static TableRow tableRow(Iterable<? extends TableRowContent> content) {
        return tr().content(content);
    }

    /**
     * @see #tr(Stream)
     */
    public static TableRow tableRow(Stream<? extends TableRowContent> content) {
        return tr().content(content);
    }

    @Override
    public TableRow copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TABLE_ROW;
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent);
    }

    private static TableRow parse(Map<String, ?> map) {
        checkType(map, Type.TABLE_ROW);
        return tr().parseRequiredContent(map, TableRowContent.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append('|');
        appendPlainTextContentJoinedWith('|', sb);
        sb.append('|');
    }
}
