package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * A table cell within a table heading row.
 * <div style="color: black; background-color: #fffae6; border-radius: 3px; padding: 16px; width: 75%;">
 * \u26A0\uFE0F <strong>WARNING</strong>: Tables are documented as supported on <strong>web</strong> and
 * <strong>desktop</strong> only. <strong>Mobile</strong> rendering support for tables may be restricted
 * or entirely unavailable.</div>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #th(TableCellContent) th}({@link Paragraph#p(String) p}("Hello world header"));
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "tableHeader",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello world header"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <table summary="">
 *     <tr>
 *         <th><p>Hello world header</p></th>
 *     </tr>
 * </table>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/tableHeader/">Node - tableHeader</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class TableHeader extends AbstractTableCellNode<TableHeader> {
    static Factory<TableHeader> FACTORY = new Factory<>(Type.TABLE_HEADER, TableHeader.class, TableHeader::parse);

    private TableHeader() {
    }

    /**
     * @return a new, empty table header. At least one content item must be added to make the table header valid.
     */
    public static TableHeader th() {
        return new TableHeader();
    }

    /**
     * @return a new table header with the given string wrapped in a {@code Paragraph} as content
     */
    public static TableHeader th(String content) {
        return th().content(content);
    }

    /**
     * @return a new table header with the given strings wrapped in a {@code Paragraph} as content
     */
    public static TableHeader th(String... content) {
        return th().content(content);
    }

    /**
     * @return a new table header with the given content
     */
    public static TableHeader th(TableCellContent content) {
        return th().content(content);
    }

    /**
     * @return a new table header with the given content
     */
    public static TableHeader th(TableCellContent... content) {
        return th().content(content);
    }

    /**
     * @return a new table header with the given content
     */
    public static TableHeader th(Iterable<? extends TableCellContent> content) {
        return th().content(content);
    }

    /**
     * @return a new table header with the given content
     */
    public static TableHeader th(Stream<? extends TableCellContent> content) {
        return th().content(content);
    }

    /**
     * @see #th()
     */
    public static TableHeader tableHeader() {
        return th();
    }

    /**
     * @see #th(TableCellContent)
     */
    public static TableHeader tableHeader(TableCellContent content) {
        return th().content(content);
    }

    /**
     * @see #th(TableCellContent[])
     */
    public static TableHeader tableHeader(TableCellContent... content) {
        return th().content(content);
    }

    /**
     * @see #th(Iterable)
     */
    public static TableHeader tableHeader(Iterable<? extends TableCellContent> content) {
        return th().content(content);
    }

    /**
     * @see #th(Stream)
     */
    public static TableHeader tableHeader(Stream<? extends TableCellContent> content) {
        return th().content(content);
    }

    @Override
    public TableHeader copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TABLE_HEADER;
    }

    /**
     * Returns {@code true} unconditionally, because this is a table header cell.
     *
     * @return {@code true} unconditionally, because this is a table header cell.
     */
    @Override
    public boolean isHeader() {
        return true;
    }

    private static TableHeader parse(Map<String, ?> map) {
        checkType(map, Type.TABLE_HEADER);
        return th().parseTableNode(map);
    }
}
