package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.TableMark;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.model.node.type.TableRowContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;

/**
 * A container for the nodes that define a table.
 * <div style="color: black; background-color: #fffae6; border-radius: 3px; padding: 16px; width: 75%;">
 * \u26A0\uFE0F <strong>WARNING</strong>: Tables are documented as supported on <strong>web</strong> and
 * <strong>desktop</strong> only. <strong>Mobile</strong> rendering support for tables may be restricted
 * or entirely unavailable.</div>
 * <p>
 * Note: The settings for numbers columns and the layout do not apply in Jira, where tables are automatically
 * displayed across the full width of the text container.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #table(Layout) table}({@link Layout#DEFAULT DEFAULT})
 *         .{@link #withoutNumberColumn() withoutNumberColumn}()
 *         .{@link #width(Number) width}(75.0)
 *         .{@link #localId(String) localId}("table-id")
 *         .{@link #tr(TableRowContent[]) tr}(
 *                 {@link TableCell#td(TableCellContent) td}({@link Paragraph#p(String) p}(" Row one, cell one")),
 *                 {@link TableCell#td(TableCellContent) td}({@link Paragraph#p(String) p}("Row one, cell two"))
 *         );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "table",
 *     "attrs": {
 *       "isNumberColumnEnabled": false,
 *       "layout": "default",
 *       "width": 75.0,
 *       "localId": "table-id"
 *     },
 *     "content": [
 *       {
 *         "type": "tableRow",
 *         "content": [
 *           {
 *             "type": "tableCell",
 *             "content": [
 *               {
 *                 "type": "paragraph",
 *                 "content": [
 *                   {
 *                     "type": "text",
 *                     "text": " Row one, cell one"
 *                   }
 *                 ]
 *               }
 *             ]
 *           },
 *           {
 *             "type": "tableCell",
 *             "content": [
 *               {
 *                 "type": "paragraph",
 *                 "content": [
 *                   {
 *                     "type": "text",
 *                     "text": "Row one, cell two"
 *                   }
 *                 ]
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <table summary="">
 *     <tr>
 *         <td><p> Row one, cell one</p></td>
 *         <td><p>Row one, cell two</p></td>
 *     </tr>
 * </table>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/table/">Node - table</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
@SuppressWarnings("UnnecessaryUnicodeEscape")
public class Table
        extends AbstractMarkedContentNode<Table, TableRow, TableMark>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent {

    static Factory<Table> FACTORY = new Factory<>(Type.TABLE, Table.class, Table::parse);

    @Nullable
    private Boolean isNumberColumnEnabled;
    @Nullable
    private Layout layout;
    @Nullable
    private Number width;
    @Nullable
    private String localId;

    private Table() {
    }

    @Override
    public Class<TableMark> markClass() {
        return TableMark.class;
    }

    /**
     * @return a new, empty table. At least one table row must be added to make it valid.
     */
    public static Table table() {
        return new Table();
    }

    /**
     * @return a new table with the given content
     */
    public static Table table(TableRow content) {
        return table().content(content);
    }

    /**
     * @return a new table with the given content
     */
    public static Table table(TableRow... content) {
        return table().content(content);
    }

    /**
     * @return a new table with the given content
     */
    public static Table table(Iterable<? extends TableRow> content) {
        return table().content(content);
    }

    /**
     * @return a new table with the given content
     */
    public static Table table(Stream<? extends TableRow> content) {
        return table().content(content);
    }

    /**
     * @return a new, empty table using the specified layout. At least one table row must be added to make it valid.
     */
    public static Table table(Layout layout) {
        return table().layout(layout);
    }

    /**
     * @return a new table with the given layout and content
     */
    public static Table table(Layout layout, TableRow content) {
        return table().layout(layout).content(content);
    }

    /**
     * @return a new table with the given layout and content
     */
    public static Table table(Layout layout, TableRow... content) {
        return table().layout(layout).content(content);
    }

    /**
     * @return a new table with the given layout and content
     */
    public static Table table(Layout layout, Iterable<? extends TableRow> content) {
        return table().layout(layout).content(content);
    }

    /**
     * @return a new table with the given layout and content
     */
    public static Table table(Layout layout, Stream<? extends TableRow> content) {
        return table().layout(layout).content(content);
    }

    /**
     * A convenience method for
     * <code>
     * {@link AbstractContentNode#content(Node) content}({@link TableRow#tableRow(TableRowContent) tableRow}(content))
     * </code>
     *
     * @param content the content to add as a single table row within the table
     * @return {@code this}
     */
    public Table tr(TableRowContent content) {
        return content(TableRow.tr(content));
    }

    /**
     * A convenience method for
     * <code>
     * {@link AbstractContentNode#content(Node) content}({@link TableRow#tableRow(TableRowContent[]) tableRow}(content))
     * </code>
     *
     * @param content the content to add as a single table row within the table
     * @return {@code this}
     */
    public Table tr(TableRowContent... content) {
        return content(TableRow.tr(content));
    }

    /**
     * A convenience method for
     * <code>
     * {@link AbstractContentNode#content(Node) content}({@link TableRow#tableRow(Iterable) tableRow}(content))
     * </code>
     *
     * @param content the content to add as a single table row within the table
     * @return {@code this}
     */
    public Table tr(Iterable<? extends TableRowContent> content) {
        return content(TableRow.tr(content));
    }

    /**
     * A convenience method for
     * <code>
     * {@link AbstractContentNode#content(Node) content}({@link TableRow#tableRow(Stream) tableRow}(content))
     * </code>
     *
     * @param content the content to add as a single table row within the table
     * @return {@code this}
     */
    public Table tr(Stream<? extends TableRowContent> content) {
        return content(TableRow.tr(content));
    }

    public boolean isNumberColumnEnabled() {
        return Boolean.TRUE.equals(isNumberColumnEnabled);
    }

    public Table isNumberColumnEnabled(@Nullable Boolean isNumberColumnEnabled) {
        this.isNumberColumnEnabled = isNumberColumnEnabled;
        return this;
    }

    public Table withNumberColumn() {
        return isNumberColumnEnabled(true);
    }

    public Table withoutNumberColumn() {
        return isNumberColumnEnabled(false);
    }

    public Optional<Layout> layout() {
        return Optional.ofNullable(layout);
    }

    public Table layout(@Nullable Layout layout) {
        this.layout = layout;
        return this;
    }

    public Optional<Number> width() {
        return Optional.ofNullable(width);
    }

    public Table width(@Nullable Number width) {
        this.width = width;
        return this;
    }

    public Optional<String> localId() {
        return Optional.ofNullable(localId);
    }

    public Table localId(@Nullable String localId) {
        this.localId = localId;
        return this;
    }

    @Override
    public Table copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.TABLE;
    }

    @Override
    protected boolean markedContentNodeEquals(Table other) {
        return layout == other.layout
                && Objects.equals(isNumberColumnEnabled, other.isNumberColumnEnabled)
                && Objects.equals(width, other.width)
                && Objects.equals(localId, other.localId);
    }

    @Override
    protected int markedContentNodeHashCode() {
        return Objects.hash(layout, isNumberColumnEnabled, width, localId);
    }

    @Override
    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
        buf.appendField("layout", layout);
        buf.appendField("isNumberColumnEnabled", isNumberColumnEnabled);
        buf.appendField("width", width);
        buf.appendField("localId", localId);
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        Map<String, ?> attrs = map()
                .addIfPresent(Attr.IS_NUMBER_COLUMN_ENABLED, isNumberColumnEnabled)
                .addMappedIfPresent(Attr.LAYOUT, layout, Layout::layout)
                .addIfPresent(Attr.WIDTH, width)
                .addIfPresent(Attr.LOCAL_ID, localId);
        return mapWithType()
                .addIf(!attrs.isEmpty(), Key.ATTRS, () -> attrs)
                .let(this::addContent)
                .let(marks::addToMap);
    }

    private static Table parse(Map<String, ?> map) {
        checkType(map, Type.TABLE);
        Table table = table()
                .parseRequiredContent(map, TableRow.class);
        getAttr(map, Attr.IS_NUMBER_COLUMN_ENABLED, Boolean.class)
                .ifPresent(table::isNumberColumnEnabled);
        getAttr(map, Attr.LAYOUT, String.class)
                .ifPresent(layout -> table.layout(Layout.PARSER.parse(layout)));
        getAttrNumber(map, Attr.WIDTH).ifPresent(table::width);
        getAttr(map, Attr.LOCAL_ID, String.class)
                .ifPresent(table::localId);
        table.parseMarks(map);
        return table;
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        appendPlainTextContentJoinedWith('\n', sb);
    }

    public enum Layout {
        DEFAULT("default"),
        FULL_WIDTH("full-width"),
        WIDE("wide");

        static final EnumParser<Layout> PARSER = new EnumParser<>(Layout.class, Layout::layout);

        private final String layout;

        Layout(String layout) {
            this.layout = layout;
        }

        public String layout() {
            return layout;
        }
    }
}
