package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.model.Element.nonNull;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;

/**
 * This node defines a status lozenge, such as is shown for the workflow status of a Jira issue or for the
 * {@code /status} macro in Confluence.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Doc#doc(DocContent[]) doc}(
 *         {@link Paragraph#p(InlineContent[]) p}(
 *                 {@link #status() status}()
 *                         .{@link Partial.NeedsColor#yellow() yellow}()
 *                         .{@link Partial.NeedsText#text(String) text}("IN PROGRESS"),
 *                 {@link Text#text(String) text}("Learn more about ADF"),
 *         )
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "version": 1,
 *     "type": "doc",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "status",
 *             "attrs": {
 *                 "color": "yellow",
 *                 "text": "IN PROGRESS"
 *             }
 *           },
 *           {
 *             "type": "text",
 *             "text": "Learn more about ADF"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <p>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <span style="border-collapse: collapse; box-sizing: border-box; background-color: #fffae6; color: #974f0c;">IN
 * PROGRESS</span> Learn more about ADF
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the status lozenge, so while it gives a vague
 * impression of what a "status lozenge" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Status
        extends AbstractNode<Status>
        implements CaptionContent, InlineContent {

    static final Factory<Status> FACTORY = new Factory<>(Type.STATUS, Status.class, Status::parse);

    private Color color;
    private String text;

    @Nullable
    private String localId;

    @Nullable
    private String style;

    private Status(Color color, String text) {
        this.color = nonNull(color, "color");
        this.text = validateText(text);
    }

    public static Status status(Color color, String text) {
        return new Status(color, text);
    }

    @CheckReturnValue
    public static Partial.NeedsColor status() {
        return new Partial.NeedsColor();
    }

    @CheckReturnValue
    public static Partial.NeedsText status(Color color) {
        return new Partial.NeedsText(color);
    }

    public static Status neutral(String text) {
        return new Status(Color.NEUTRAL, text);
    }

    public static Status purple(String text) {
        return new Status(Color.PURPLE, text);
    }

    public static Status blue(String text) {
        return new Status(Color.BLUE, text);
    }

    public static Status red(String text) {
        return new Status(Color.RED, text);
    }

    public static Status yellow(String text) {
        return new Status(Color.YELLOW, text);
    }

    public static Status green(String text) {
        return new Status(Color.GREEN, text);
    }

    public Color color() {
        return color;
    }

    public Status color(Color color) {
        this.color = nonNull(color, "color");
        return this;
    }

    public Status neutral() {
        return color(Color.NEUTRAL);
    }

    public Status purple() {
        return color(Color.PURPLE);
    }

    public Status blue() {
        return color(Color.BLUE);
    }

    public Status red() {
        return color(Color.RED);
    }

    public Status yellow() {
        return color(Color.YELLOW);
    }

    public Status green() {
        return color(Color.GREEN);
    }

    public Status text(String text) {
        this.text = validateText(text);
        return this;
    }

    public String text() {
        return text;
    }

    public Status localId(String localId) {
        this.localId = localId;
        return this;
    }

    public Status style(String style) {
        this.style = style;
        return this;
    }

    public Optional<String> localId() {
        return Optional.ofNullable(localId);
    }

    public Optional<String> style() {
        return Optional.ofNullable(style);
    }

    @Override
    public Status copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.STATUS;
    }

    @Override
    public void validate() {
    }

    @Override
    protected boolean nodeEquals(Status other) {
        return color == other.color
                && text.equals(other.text)
                && Objects.equals(localId, other.localId)
                && Objects.equals(style, other.style);
    }

    @Override
    protected int nodeHashCode() {
        return Objects.hash(color, text, localId, style);
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendTextField(text);
        buf.appendField("color", color);
        buf.appendField("localId", localId);
        buf.appendField("style", style);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.COLOR, color.color)
                        .add(Attr.TEXT, text)
                        .addIfPresent(Attr.LOCAL_ID, localId)
                        .addIfPresent(Attr.STYLE, style)
                );
    }

    private static Status parse(Map<String, ?> map) {
        Color color = Color.PARSER.parse(getAttrOrThrow(map, Attr.COLOR));
        String text = getAttrOrThrow(map, Attr.TEXT);
        Status status = new Status(color, text);
        getAttr(map, Attr.LOCAL_ID, String.class).ifPresent(status::localId);
        getAttr(map, Attr.STYLE, String.class).ifPresent(status::style);
        return status;
    }

    private static String validateText(String text) {
        return nonEmpty(text, "text");
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append("[ ")
                .append(text.toUpperCase(Locale.ROOT))
                .append(" ]");
    }

    /**
     * The color used for the status lozenge.
     */
    public enum Color {
        NEUTRAL("neutral"),
        PURPLE("purple"),
        BLUE("blue"),
        RED("red"),
        YELLOW("yellow"),
        GREEN("green");

        static final EnumParser<Color> PARSER = new EnumParser<>(Color.class, Color::color);

        private final String color;

        Color(String color) {
            this.color = color;
        }

        public String color() {
            return color;
        }
    }


    /**
     * Types that represent a partially constructed {@link Status status}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code status} still needs its {@code color} attribute set.
         */
        class NeedsColor {
            NeedsColor() {
            }

            @CheckReturnValue
            public NeedsText color(Color color) {
                return new NeedsText(color);
            }

            @CheckReturnValue
            public NeedsText neutral() {
                return new NeedsText(Color.NEUTRAL);
            }

            @CheckReturnValue
            public NeedsText purple() {
                return new NeedsText(Color.PURPLE);
            }

            @CheckReturnValue
            public NeedsText blue() {
                return new NeedsText(Color.BLUE);
            }

            @CheckReturnValue
            public NeedsText red() {
                return new NeedsText(Color.RED);
            }

            @CheckReturnValue
            public NeedsText yellow() {
                return new NeedsText(Color.YELLOW);
            }

            @CheckReturnValue
            public NeedsText green() {
                return new NeedsText(Color.GREEN);
            }
        }

        /**
         * This partially constructed {@code status} still needs its {@code text} attribute set.
         */
        class NeedsText {
            private final Color color;

            public NeedsText(Color color) {
                this.color = nonNull(color, "color");
            }

            public Status text(String text) {
                return new Status(color, text);
            }
        }
    }
}
