package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;

/**
 * Provides placeholder text for something.
 * <p>
 * Facts:
 * <ul>
 *     <li>This seems to render an empty {@code <span></span>}</li>
 *     <li>The unpublished documentation for it just says it is a container.</li>
 *     <li>The unpublished documentation includes an optional {@code content} array</li>
 *     <li>The JSON schema, however, does not allow any {@code content} within a {@code placeholder_node}</li>
 * </ul>
 * What are we to make of this? Your humble library author speculates that it is intended to be dynamically
 * replaced with some other content after the initial rendering and that the editor team never fully
 * implemented this feature. It may be that the {@link InlineExtension inlineExtension} has made this
 * node type obsolete. These are just wild guesses, however.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Placeholder
        extends AbstractNode<Placeholder>
        implements CaptionContent, InlineContent {

    static final Factory<Placeholder> FACTORY = new Factory<>(Type.PLACEHOLDER, Placeholder.class, Placeholder::parse);

    private String text;

    private Placeholder(String text) {
        this.text = validateText(text);
    }

    public Partial.NeedsText placeholder() {
        return new Partial.NeedsText();
    }

    public static Placeholder placeholder(String text) {
        return new Placeholder(text);
    }

    public Placeholder text(String text) {
        this.text = validateText(text);
        return this;
    }

    public String text() {
        return text;
    }

    @Override
    public Placeholder copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.PLACEHOLDER;
    }

    @Override
    public void validate() {
    }

    @Override
    protected int nodeHashCode() {
        return text.hashCode();
    }

    @Override
    protected boolean nodeEquals(Placeholder other) {
        return text.equals(other.text);
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        buf.appendTextField(text);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Key.TEXT, text)
                );
    }

    private static Placeholder parse(Map<String, ?> map) {
        String text = getAttrOrThrow(map, Attr.TEXT);
        return new Placeholder(text);
    }

    private static String validateText(String text) {
        return nonEmpty(text, "text");
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append(text);
    }

    /**
     * Types that represent a partially constructed {@link Placeholder placeholder}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code placeholder} still needs its {@code text} attribute set.
         */
        class NeedsText {
            NeedsText() {
            }

            public Placeholder text(String text) {
                return new Placeholder(text);
            }
        }
    }
}
