package com.atlassian.adf.model.node;

import com.atlassian.adf.model.mark.Alignment;
import com.atlassian.adf.model.mark.Indentation;
import com.atlassian.adf.model.mark.type.ParagraphMark;
import com.atlassian.adf.model.mark.type.PositionMark;
import com.atlassian.adf.model.mark.type.TextMark;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.ListItemContent;
import com.atlassian.adf.model.node.type.NestedExpandContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.PanelContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import java.util.Arrays;
import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * A container for a block of formatted text delineated by a carriage return.
 * It is equivalent to an HTML {@code <p>} tag.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(String) p}("Hello world");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello world"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello world</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/paragraph/">Node - paragraph</a>
 */
public class Paragraph
        extends AbstractMarkedContentNode<Paragraph, InlineContent, ParagraphMark>
        implements DocContent, LayoutColumnContent, ListItemContent, NestedExpandContent, NonNestableBlockContent,
        PanelContent, TableCellContent {

    static Factory<Paragraph> FACTORY = new Factory<>(Type.PARAGRAPH, Paragraph.class, Paragraph::parse);

    private Paragraph() {
    }

    /**
     * @return a new, empty paragraph
     */
    public static Paragraph p() {
        return new Paragraph();
    }

    /**
     * @return a new paragraph with the given content
     */
    public static Paragraph p(String content) {
        return p(Text.text(content));
    }

    /**
     * Creates a new paragraph with the given content, all within that single paragraph. This is so that
     * a long paragraph can be nicely split over multiple lines without having to concatenate them manually.
     * Each is assigned its own {@link Text} node, but they are all assigned as content within the same
     * containing paragraph.
     *
     * @return a new paragraph with the given content
     */
    public static Paragraph p(String... content) {
        return p(Arrays.stream(content).map(Text::text));
    }

    /**
     * @return a new paragraph with the given content
     */
    public static Paragraph p(InlineContent content) {
        return p().content(content);
    }

    /**
     * @return a new paragraph with the given content
     */
    public static Paragraph p(InlineContent... content) {
        return p().content(content);
    }

    /**
     * @return a new paragraph with the given content
     */
    public static Paragraph p(Iterable<? extends InlineContent> content) {
        return p().content(content);
    }

    /**
     * @return a new paragraph with the given content
     */
    public static Paragraph p(Stream<? extends InlineContent> content) {
        return p().content(content);
    }

    /**
     * @see #p()
     */
    public static Paragraph paragraph() {
        return new Paragraph();
    }

    /**
     * @see #p(String)
     */
    public static Paragraph paragraph(String content) {
        return paragraph(Text.text(content));
    }

    /**
     * @see #p(String[])
     */
    public static Paragraph paragraph(String... content) {
        return paragraph(Arrays.stream(content).map(Text::text));
    }

    /**
     * @see #p(InlineContent)
     */
    public static Paragraph paragraph(InlineContent content) {
        return paragraph().content(content);
    }

    /**
     * @see #p(InlineContent[])
     */
    public static Paragraph paragraph(InlineContent... content) {
        return paragraph().content(content);
    }

    /**
     * @see #p(Iterable)
     */
    public static Paragraph paragraph(Iterable<? extends InlineContent> content) {
        return paragraph().content(content);
    }

    /**
     * @see #p(Stream)
     */
    public static Paragraph paragraph(Stream<? extends InlineContent> content) {
        return paragraph().content(content);
    }

    /**
     * Convenience method for appending an ordinary text node to this paragraph.
     * <p>
     * This is equivalent to:
     * <pre><code>
     * {@link AbstractContentNode#content(Node) content}({@link Text#text(String) text}(content))
     * </code></pre>
     *
     * @param content the text to add as an ordinary text node
     * @return {@code this}
     */
    public Paragraph text(String content) {
        return content(Text.text(content));
    }

    /**
     * Convenience method for appending an text node with a single mark to this paragraph.
     * <p>
     * This is equivalent to:
     * <pre><code>
     * {@link AbstractContentNode#content(Node) content}(
     *         {@link Text#text(String) text}(content)
     *                 .{@link Text#mark(TextMark) mark}(mark)
     * );
     * </code></pre>
     *
     * @param content the text to add as an ordinary text node
     * @return {@code this}
     */
    public Paragraph text(String content, TextMark mark) {
        return content(Text.text(content, mark));
    }

    /**
     * Convenience method for appending an text node with multiple marks to this paragraph.
     * <p>
     * This is equivalent to:
     * <pre><code>
     * {@link AbstractContentNode#content(Node) content}({@link Text#text(String, TextMark[]) text}(content, marks));
     * </code></pre>
     *
     * @param content the text to add as an ordinary text node
     * @return {@code this}
     */
    public Paragraph text(String content, TextMark... marks) {
        return content(Text.text(content, marks));
    }

    /**
     * Convenience method for appending multiple ordinary text nodes to this paragraph.
     * <p>
     * This is equivalent to:
     * <pre><code>
     * {@link AbstractContentNode#content(Node) content}({@link Text#text(String[]) text}(content))
     * </code></pre>
     *
     * @param content the text to add as an ordinary text node
     * @return {@code this}
     */
    public Paragraph text(String... content) {
        return content(Text.text(content));
    }

    public Paragraph alignment(Alignment alignment) {
        return mark(alignment);
    }

    public Paragraph center() {
        return mark(Alignment.center());
    }

    public Paragraph end() {
        return mark(Alignment.end());
    }

    public Paragraph indentation(Indentation indentation) {
        return mark(indentation);
    }

    public Paragraph indentation(int level) {
        return mark(Indentation.indentation(level));
    }

    @Override
    public Paragraph mark(ParagraphMark mark) {
        super.mark(mark);
        if (mark instanceof PositionMark) restrictPositionMarks((PositionMark) mark);
        return this;
    }

    @Override
    public Class<ParagraphMark> markClass() {
        return ParagraphMark.class;
    }

    @Override
    public Paragraph copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.PARAGRAPH;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .let(this::addContentIfPresent)
                .let(marks::addToMap);
    }

    private static Paragraph parse(Map<String, ?> map) {
        checkType(map, Type.PARAGRAPH);
        return p()
                .parseOptionalContent(map, InlineContent.class)
                .parseMarks(map);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        appendPlainTextInlineContent(sb);
    }

    void disableMarks(ContentNode<?, ? super Paragraph> parent) {
        marks.disable(parent.elementType());
    }

    private void restrictPositionMarks(PositionMark markThatWasThereFirst) {
        Class<? extends PositionMark> allowedClass = markThatWasThereFirst.getClass();
        String reason = "Only one PositionMark is permitted, and '" + markThatWasThereFirst.elementType() +
                "' is already present";
        marks.addRule(
                mark -> allowedClass.isInstance(mark) || !(mark instanceof PositionMark),
                reason
        );
    }

    // FIXME? Per the JSON schema, "paragraph" nodes are not allowed to use "indentation" marks in
    // "tableCell", "tableHeader", or "layoutColumn" nodes even though "heading" nodes are allowed to do that.
    void disableIndentation(ContentNode<?, ? super Paragraph> parent) {
        marks.rejectInstancesOf(Indentation.class, parent.elementType());
    }
}
