package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Element;
import com.atlassian.adf.model.ex.AdfException;

import java.util.Map;

/**
 * A node in the Atlassian Document Format describes an element to be rendered as
 * rich content. In the JSON transport format, each node is represented as an object
 * with a mandatory {@link Key#TYPE type} property that identifies what kind of node
 * it represents.
 * <p>
 * <strong>WARNING</strong>: Some node types are not supported on mobile. Consider your
 * audience when building content in an automated fashion and whether you have provided
 * enough information for those environments where, for example, tables may not be rendered.
 */
public interface Node extends Element {
    /**
     * Parses a node of unknown type and without any known context for it.
     * If the node happens to be a {@code doc}, then this is equivalent to calling {@link Doc#parse(Map)} directly.
     * Otherwise, the ADF node is parsed as an isolated fragment with no restrictions on the type of node it may be.
     *
     * @param map the ADF content to parse.
     * @return the parsed node
     * @throws AdfException if the content cannot be parsed
     */
    static Node parse(Map<String, ?> map) {
        return Type.DOC.equals(map.get(Key.TYPE))
                ? Doc.parse(map)
                : NodeParserSupport.getNodeOfType(Node.class, map, "(fragment)");
    }

    /**
     * Parses a node of unknown type and without any known context for it.
     * If the node happens to be a {@code doc} and the {@code requiredClass} is appropriate, then this is
     * equivalent to calling {@link Doc#parse(Map)} directly.
     * Otherwise, the ADF node is parsed as an isolated fragment with the only restriction being that the
     * returned node must be assignable to {@code requiredClass}.
     *
     * @param requiredClass a return type to restrict the types of nodes that may be returned
     * @param map           the ADF content to parse.
     * @return the parsed node
     * @throws AdfException if the content cannot be parsed or is not assignable to {@code requiredClass}
     */
    static <T extends Node> T parse(Class<T> requiredClass, Map<String, ?> map) {
        return (Type.DOC.equals(map.get(Key.TYPE)) && requiredClass.isAssignableFrom(Doc.class))
                ? requiredClass.cast(Doc.parse(map))
                : NodeParserSupport.getNodeOfType(requiredClass, map, "(fragment)");
    }

    @Override
    Node copy();

    /**
     * Renders this node as plain-text suitable for viewing by end users.
     * Note that this is distinct from the more informative {@code toString()} value, which is meant
     * for debugging and logging, only.
     */
    String toPlainText();

    /**
     * Renders this node as plain-text suitable for viewing by end users.
     * This is equivalent to calling {@link #toPlainText()} and appending the result to the given
     * buffer, except that it may be slightly more efficient, since it will write directly to the
     * existing buffer instead of using a temporary buffer and having to make a copy of the result.
     *
     * @param sb where to write the result
     */
    void appendPlainText(StringBuilder sb);

    /**
     * Constants used as property names in nodes.
     */
    interface Key extends Element.Key {
        /**
         * This field contains an array of child nodes.
         * These are generally restricted to a specific node type, which this library enforces
         * through the Java type system wherever possible.
         *
         * @see com.atlassian.adf.model.node.type.ContentNode
         */
        String CONTENT = "content";

        /**
         * This field contains an array of marks that decorate the node.
         * These generally affect the node's presentation or (in the case of extension nodes) help
         * connect the extension
         */
        String MARKS = "marks";

        /**
         * This field contains a text value to be displayed, either as normal text content or
         * as a placeholder for something else.
         */
        String TEXT = "text";

        /**
         * This field is only used by the top-level {@link Doc doc} node, to indicate the current ADF
         * document structure version.
         */
        String VERSION = "version";
    }

    /**
     * Constants used as the {@link Element.Key#TYPE type} names for the various ADF nodes.
     */
    interface Type {
        String BLOCK_CARD = "blockCard";
        String BLOCKQUOTE = "blockquote";
        String BODIED_EXTENSION = "bodiedExtension";
        String BULLET_LIST = "bulletList";
        String CAPTION = "caption";
        String CODE_BLOCK = "codeBlock";
        String DATE = "date";
        String DECISION_ITEM = "decisionItem";
        String DECISION_LIST = "decisionList";
        String DOC = "doc";
        String EMBED_CARD = "embedCard";
        String EMOJI = "emoji";
        String EXPAND = "expand";
        String EXTENSION = "extension";
        String HARD_BREAK = "hardBreak";
        String HEADING = "heading";
        String INLINE_CARD = "inlineCard";
        String INLINE_EXTENSION = "inlineExtension";
        String LAYOUT_COLUMN = "layoutColumn";
        String LAYOUT_SECTION = "layoutSection";
        String LIST_ITEM = "listItem";
        String MEDIA = "media";
        String MEDIA_GROUP = "mediaGroup";
        String MEDIA_INLINE = "mediaInline";
        String MEDIA_SINGLE = "mediaSingle";
        String MENTION = "mention";
        String NESTED_EXPAND = "nestedExpand";
        String ORDERED_LIST = "orderedList";
        String PANEL = "panel";
        String PARAGRAPH = "paragraph";
        String PLACEHOLDER = "placeholder";
        String RULE = "rule";
        String STATUS = "status";
        String TABLE = "table";
        String TABLE_CELL = "tableCell";
        String TABLE_HEADER = "tableHeader";
        String TABLE_ROW = "tableRow";
        String TASK_ITEM = "taskItem";
        String TASK_LIST = "taskList";
        String TEXT = "text";
    }

    /**
     * Constants used as the {@link Element.Key#ATTRS attribute} names in nodes.
     */
    interface Attr extends Element.Attr {
        String ACCESS_LEVEL = "accessLevel";
        String ALT = "alt";
        String BACKGROUND = "background";
        String COLSPAN = "colspan";
        String COLWIDTH = "colwidth";
        String DATA = "data";
        String EXTENSION_KEY = "extensionKey";
        String EXTENSION_TYPE = "extensionType";
        String HEIGHT = "height";
        String IS_NUMBER_COLUMN_ENABLED = "isNumberColumnEnabled";
        String LANGUAGE = "language";
        String LAYOUT = "layout";
        String ORDER = "order";
        String ORIGINAL_HEIGHT = "originalHeight";
        String ORIGINAL_WIDTH = "originalWidth";
        String PANEL_COLOR = "panelColor";
        String PANEL_ICON = "panelIcon";
        String PANEL_ICON_ID = "panelIconId";
        String PANEL_ICON_TEXT = "panelIconText";
        String PANEL_TYPE = "panelType";
        String PARAMETERS = "parameters";
        String ROWSPAN = "rowspan";
        String SHORT_NAME = "shortName";
        String STATE = "state";
        String STYLE = "style";
        String TIMESTAMP = "timestamp";
        String TYPE = "type";
        String URL = "url";
        String USER_TYPE = "userType";
        String WIDTH = "width";
        String WIDTH_TYPE = "widthType";
    }

}
