package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.NestedExpandContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;

/**
 * A {@code nestedExpand} node serves the same purpose as {@link Expand expand}, except that it is used
 * inside of a {@code tableCell} or {@code tableHeader} instead of directly inside a {@link Doc doc}.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class NestedExpand
        extends AbstractContentNode<NestedExpand, NestedExpandContent>
        implements TableCellContent {

    static final Factory<NestedExpand> FACTORY = new Factory<>(
            Type.NESTED_EXPAND,
            NestedExpand.class,
            NestedExpand::parse
    );

    private NestedExpand() {
    }

    public static NestedExpand nestedExpand() {
        return new NestedExpand();
    }

    public static NestedExpand nestedExpand(NestedExpandContent content) {
        return nestedExpand().content(content);
    }

    public static NestedExpand nestedExpand(NestedExpandContent... content) {
        return nestedExpand().content(content);
    }

    public static NestedExpand nestedExpand(Iterable<? extends NestedExpandContent> content) {
        return nestedExpand().content(content);
    }

    public static NestedExpand nestedExpand(Stream<? extends NestedExpandContent> content) {
        return nestedExpand().content(content);
    }

    @Nullable
    private String title;

    public Optional<String> title() {
        return Optional.ofNullable(title);
    }

    public NestedExpand title(@Nullable String title) {
        this.title = title;
        return this;
    }

    @Override
    public NestedExpand copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.NESTED_EXPAND;
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    protected boolean contentNodeEquals(NestedExpand other) {
        return Objects.equals(title, other.title);
    }

    @Override
    protected int contentNodeHashCode() {
        return Objects.hashCode(title);
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("title", title);
    }

    @Override
    protected void validateContentNodeForAppend(NestedExpandContent node) {
        if (node instanceof Paragraph) {
            ((Paragraph) node).disableMarks(this);
        } else if (node instanceof Heading) {
            ((Heading) node).disableMarks(this);
        }
    }

    // FIXME? Per the JSON schema, "nestedExpand" nodes are required to provide an "attrs" map even if it is empty.
    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .add(Key.ATTRS, map().addIfPresent(Attr.TITLE, title));
    }

    void disableMarks(ContentNode<?, ? super NestedExpand> parent) {
        // NestedExpand can't have any marks anyway, but the schema explicitly calls out this restriction
        // in the definition for table_cell_content, so...
    }

    private static NestedExpand parse(Map<String, ?> map) {
        checkType(map, Type.NESTED_EXPAND);
        NestedExpand nestedExpand = new NestedExpand()
                .parseRequiredContent(map, NestedExpandContent.class);
        getOrThrow(map, Key.ATTRS);
        getAttr(map, Attr.TITLE, String.class).ifPresent(nestedExpand::title);
        return nestedExpand;
    }
}
