package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.MediaException;
import com.atlassian.adf.model.mark.Link;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.type.MediaInlineMark;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.Factory;
import com.atlassian.adf.util.FieldMap;

import javax.annotation.Nullable;
import java.net.URL;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.model.Element.nonNull;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;

/**
 * The {@code mediaInline} node type works just like {@link Media.MediaType#FILE file} and
 * {@link Media.MediaType#LINK link} types of {@link Media media} nodes, except that instead of
 * being placed inside the {@link MediaGroup mediaGroup} and {@link MediaSingle mediaSingle} block
 * nodes, they are used in places that {@link InlineContent inline content} is permitted, such as
 * within a {@link Paragraph paragraph}.
 * <p>
 * Note that while the media {@code collection} is being phased out, it is currently still required
 * by the schema. This node allows the caller to fail to set it or set it to {@code null}, but will
 * substitute the empty string {@code ""} instead, for the time being.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
@SuppressWarnings("UnusedReturnValue")
public class MediaInline
        extends AbstractMarkedNode<MediaInline, MediaInlineMark>
        implements InlineContent {

    static Factory<MediaInline> FACTORY = new Factory<>(Type.MEDIA_INLINE, MediaInline.class, MediaInline::parse);

    private String id;
    private String collection = "";

    @Nullable
    private MediaType mediaType;
    @Nullable
    private String occurrenceKey;
    @Nullable
    private Number width;
    @Nullable
    private Number height;
    @Nullable
    private String alt;
    @Nullable
    private Map<String, ?> data;

    private MediaInline(String id) {
        this.id = validateId(id);
    }

    public MediaInline link(@Nullable Link link) {
        marks.remove(Mark.Type.LINK);
        if (link != null) marks.add(link);
        return this;
    }

    public MediaInline link(@Nullable String href) {
        return link((href != null) ? Link.link(href) : null);
    }

    public MediaInline data(@Nullable Map<String, ?> data) {
        this.data = (data != null) ? FieldMap.immutableCopy(data) : null;
        return this;
    }

    /**
     * Returns the mediaInline node's ID.
     *
     * @return the mediaInline node's ID.
     */
    public String id() {
        return id;
    }

    public MediaInline id(String id) {
        this.id = validateId(id);
        return this;
    }

    /**
     *
     * @return the mediaInline node's collection ID.
     */
    public String collection() {
        return collection;
    }

    public MediaInline collection(@Nullable String collection) {
        this.collection = validateCollection(collection);
        return this;
    }

    /**
     * Returns the mediaInline node's occurrence key, if set.
     *
     * @return the mediaInline node's occurrence key, or {@code empty()} if not set.
     */
    public Optional<String> occurrenceKey() {
        return Optional.ofNullable(occurrenceKey);
    }

    /**
     * Sets the occurrence key for this mediaInline item.
     * Although this attribute is optional, it must be set to enable deletion of files from a collection.
     *
     * @param occurrenceKey the occurrence key value
     * @return {@code this}
     */
    public MediaInline occurrenceKey(@Nullable String occurrenceKey) {
        this.occurrenceKey = validateOccurrenceKey(occurrenceKey);
        return this;
    }

    public static Partial.NeedsId mediaInline() {
        return new Partial.NeedsId(null);
    }

    public static MediaInline mediaInline(String id) {
        return new MediaInline(id);
    }

    public static Partial.NeedsId mediaInlineFile() {
        return new Partial.NeedsId(MediaType.FILE);
    }

    public static MediaInline mediaInlineFile(String id) {
        return mediaInline(id).file();
    }

    public static Partial.NeedsId mediaInlineLink() {
        return new Partial.NeedsId(MediaType.LINK);
    }

    public static MediaInline mediaInlineLink(String id) {
        return mediaInline(id).link();
    }

    public MediaInline file() {
        return mediaType(MediaType.FILE);
    }

    public MediaInline link() {
        return mediaType(MediaType.LINK);
    }

    public MediaInline mediaType(@Nullable String mediaType) {
        return mediaType(MediaType.PARSER.parseAllowNull(mediaType));
    }

    public MediaInline mediaType(@Nullable MediaType mediaType) {
        this.mediaType = mediaType;
        return this;
    }

    /**
     * Returns the mediaInline node's display width, if set.
     *
     * @return the mediaInline node's display width, or {@code empty()} if not set.
     */
    public Optional<Number> width() {
        return Optional.ofNullable(width);
    }

    /**
     * Sets the width of the media.
     *
     * @param width the display width of the mediaInline item, in pixels; must be positive
     * @return {@code this}
     */
    public MediaInline width(Number width) {
        nonNull(width, "width");
        if (width.doubleValue() <= 0.0) {
            throw new MediaException.WidthMustBePositive(width);
        }
        this.width = width;
        return this;
    }

    /**
     * Returns the mediaInline node's display height, if set.
     *
     * @return the mediaInline node's display height, or {@code empty()} if not set.
     */
    public Optional<Number> height() {
        return Optional.ofNullable(height);
    }

    /**
     * Sets the height of the media.
     *
     * @param height the display height of the mediaInline item, in pixels; must be positive
     * @return {@code this}
     */
    public MediaInline height(Number height) {
        nonNull(height, "height");
        if (height.doubleValue() <= 0.0) {
            throw new MediaException.HeightMustBePositive(height);
        }
        this.height = height;
        return this;
    }

    /**
     * Sets the width and height of the media.
     * <p>
     * This convenience method is exactly equivalent to calling
     * <code>{@link #width(Number) width}(width).{@link #height(Number) height}(height)</code>.
     *
     * @param width  as for {@link #width(Number)}
     * @param height as for {@link #height(Number)}
     * @return {@code this}
     */
    public MediaInline size(Number width, Number height) {
        return width(width).height(height);
    }

    public MediaInline alt(@Nullable String alt) {
        this.alt = alt;
        return this;
    }

    public MediaInline linkMark(@Nullable Link link) {
        return link(link);
    }

    public MediaInline linkMark(@Nullable URL url) {
        return link((url != null) ? Link.link(url) : null);
    }

    public MediaInline linkMark(@Nullable String href) {
        return link((href != null) ? Link.link(href) : null);
    }

    public Optional<Link> linkMark() {
        return marks.get(Mark.Type.LINK)
                .map(Link.class::cast);
    }

    public Optional<Map<String, ?>> data() {
        return Optional.ofNullable(data);
    }

    /**
     * Creates a new {@code mediaInline} node with the given media ID and collection ID.
     *
     * @param id         the Media Services ID used for querying the media services API to retrieve metadata,
     *                   such as the filename. Consumers of the document should always fetch fresh metadata
     *                   using the Media API rather than cache it locally.
     * @param collection the MediaInline Services Collection name for the media
     * @return the new {@code media} node
     */
    public static MediaInline mediaInline(String id, @Nullable String collection) {
        return new MediaInline(id).collection(collection);
    }

    /**
     * Creates a new {@code mediaInline} node with the given media ID, media type, and collection ID.
     *
     * @param id         the Media Services ID used for querying the media services API to retrieve metadata,
     *                   such as the filename. Consumers of the document should always fetch fresh metadata
     *                   using the Media API rather than cache it locally.
     * @param mediaType  the media type to assign
     * @param collection the MediaInline Services Collection name for the media
     * @return the new {@code media} node
     */
    public static MediaInline mediaInline(String id, @Nullable MediaType mediaType, @Nullable String collection) {
        MediaInline mediaInline = new MediaInline(id).collection(collection);
        if (mediaType != null) {
            mediaInline.mediaType(mediaType);
        }
        return mediaInline;
    }

    @Override
    public MediaInline copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.MEDIA_INLINE;
    }

    @Override
    public Class<MediaInlineMark> markClass() {
        return MediaInlineMark.class;
    }

    @Override
    protected boolean markedNodeEquals(MediaInline other) {
        return id.equals(other.id)
                && collection.equals(other.collection)
                && Objects.equals(occurrenceKey, other.occurrenceKey)
                && Objects.equals(mediaType, other.mediaType)
                && Objects.equals(alt, other.alt)
                && numberEq(width, other.width)
                && numberEq(height, other.height);
    }

    @Override
    protected int markedNodeHashCode() {
        return Objects.hash(id, collection, occurrenceKey, mediaType, numberHash(width), numberHash(height), alt);
    }

    @Override
    protected void appendMarkedNodeFields(ToStringHelper buf) {
        buf.appendField("id", id);
        buf.appendField("collection", collection);
        buf.appendField("occurrenceKey", occurrenceKey);
        buf.appendField("mediaType", mediaType);
        buf.appendField("width", width);
        buf.appendField("height", height);
        buf.appendField("alt", alt);
        buf.appendField("data", data);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.ID, id)
                        .add(Attr.COLLECTION, collection)
                        .addMappedIfPresent(Attr.TYPE, mediaType, MediaType::mediaType)
                        .addIfPresent(Attr.OCCURRENCE_KEY, occurrenceKey)
                        .addIfPresent(Attr.WIDTH, width)
                        .addIfPresent(Attr.HEIGHT, height)
                        .addIfPresent(Attr.ALT, alt)
                        .addIfPresent(Attr.DATA, data)
                )
                .let(marks::addToMap);
    }

    private static MediaInline parse(Map<String, ?> map) {
        checkType(map, Type.MEDIA_INLINE);
        String id = getAttrOrThrow(map, Attr.ID);
        String collection = getAttr(map, Attr.COLLECTION, String.class).orElse(null);
        MediaInline mediaInline = new MediaInline(id).collection(collection);
        getAttrNumber(map, Attr.WIDTH).ifPresent(mediaInline::width);
        getAttrNumber(map, Attr.HEIGHT).ifPresent(mediaInline::height);
        getAttr(map, Attr.TYPE, String.class).ifPresent(mediaInline::mediaType);
        getAttr(map, Attr.ALT, String.class).ifPresent(mediaInline::alt);
        getAttr(map, Attr.OCCURRENCE_KEY, String.class).ifPresent(mediaInline::occurrenceKey);
        getAttr(map, Attr.DATA).ifPresent(data -> mediaInline.data(unsafeCast(data)));
        return mediaInline.parseMarks(map);
    }

    private static String validateId(String id) {
        return nonEmpty(id, "id");
    }

    private static String validateCollection(@Nullable String collection) {
        return (collection != null) ? collection : "";
    }

    @Nullable
    private static String validateOccurrenceKey(@Nullable String occurrenceKey) {
        return (occurrenceKey == null || !occurrenceKey.isEmpty()) ? occurrenceKey : null;
    }


    /**
     * Types that represent a partially constructed {@link MediaInline mediaInline}.
     */
    public interface Partial {
        /**
         * The media type of this partially constructed {@code mediaInline} can still be changed.
         */
        abstract class CanSetMediaType<T extends CanSetMediaType<T>> {
            @Nullable
            protected MediaType mediaType;

            CanSetMediaType(@Nullable MediaType mediaType) {
                this.mediaType = mediaType;
            }

            public T file() {
                return mediaType(MediaType.FILE);
            }

            public T link() {
                return mediaType(MediaType.LINK);
            }

            public T mediaType(String mediaType) {
                return mediaType(MediaType.PARSER.parse(mediaType));
            }

            public T mediaType(MediaType mediaType) {
                this.mediaType = mediaType;
                return unsafeCast(this);
            }
        }

        /**
         * This partially constructed {@code mediaInline} still needs its media {@code id} attribute set.
         */
        class NeedsId extends CanSetMediaType<NeedsId> {
            NeedsId() {
                this(null);
            }

            NeedsId(@Nullable MediaType mediaType) {
                super(mediaType);
            }

            public MediaInline id(String id) {
                return new MediaInline(id).mediaType(mediaType);
            }
        }
    }


    public enum MediaType {
        FILE("file"),
        LINK("link");

        static final EnumParser<MediaType> PARSER = new EnumParser<>(MediaType.class, MediaType::mediaType);

        private final String mediaType;

        MediaType(String mediaType) {
            this.mediaType = mediaType;
        }

        public String mediaType() {
            return mediaType;
        }
    }
}