package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.node.LayoutColumnException;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrNumberOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Declares one of the 2 to 3 columns in a {@link LayoutSection layoutSection}.
 * <h2>Example</h2>
 * See {@link LayoutSection layoutSection}.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class LayoutColumn
        extends AbstractContentNode<LayoutColumn, LayoutColumnContent> {

    static final Factory<LayoutColumn> FACTORY = new Factory<>(
            Type.LAYOUT_COLUMN,
            LayoutColumn.class,
            LayoutColumn::parse
    );

    private Number width;

    private LayoutColumn(Number width) {
        this.width = validateWidth(width);
    }

    public static LayoutColumn layoutColumn(Number width) {
        return new LayoutColumn(width);
    }

    public LayoutColumn width(Number width) {
        this.width = validateWidth(width);
        return this;
    }

    public Number width() {
        return width;
    }

    @Override
    public LayoutColumn copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.LAYOUT_COLUMN;
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    @Override
    protected void validateContentNodeForAppend(LayoutColumnContent node) {
        if (node instanceof CodeBlock) {
            ((CodeBlock) node).disableMarks(this);
        } else if (node instanceof Expand) {
            ((Expand) node).disableMarks(this);
        }
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.WIDTH, width)
                )
                .let(this::addContent);
    }

    private static LayoutColumn parse(Map<String, ?> map) {
        checkType(map, Type.LAYOUT_COLUMN);
        Number width = getAttrNumberOrThrow(map, Attr.WIDTH);
        return new LayoutColumn(width)
                .parseRequiredContent(map, LayoutColumnContent.class);
    }

    private static Number validateWidth(Number width) {
        requireNonNull(width, "width");
        double widthAsDouble = width.doubleValue();
        if (widthAsDouble < 0.0 || widthAsDouble > 100.0) {
            throw new LayoutColumnException.InvalidWidth(width);
        }
        return width;
    }
}
