package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import java.net.URI;
import java.net.URL;
import java.util.Map;

import static com.atlassian.adf.model.node.unsupported.UnsupportedNode.plainTextFallback;
import static com.atlassian.adf.util.ParserSupport.checkType;

/**
 * An Atlassian link card with a type icon and content description derived from the link.  The
 * {@code inlineCard} node must specify either a URL or a <a href="https://json-ld.org/">JSON-LD</a>}
 * representation of the link, but not both.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #inlineCard(String) inlineCard}("https://atlassian.com");
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "inlineCard",
 *     "attrs": {
 *       "url": "https://atlassian.com"
 *     }
 *   }
 * }</pre>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/inlineCard/">Node - inlineCard</a>
 * @see <a href="https://json-ld.org/">JSON-LD</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class InlineCard
        extends AbstractCardNode<InlineCard>
        implements CaptionContent, InlineContent {

    static Factory<InlineCard> FACTORY = new Factory<>(Type.INLINE_CARD, InlineCard.class, InlineCard::parse);

    private InlineCard(UrlOrData urlOrData) {
        super(urlOrData);
    }

    /**
     * Creates a partially constructed inline card that requires its URL or data to be set before it is valid.
     *
     * @return a partially constructed inline card
     */
    @CheckReturnValue
    public static Partial.NeedsUrlOrData<InlineCard> inlineCard() {
        return new Partial.NeedsUrlOrData<>(InlineCard::new);
    }

    /**
     * Creates an inline card for the linked content with the given URI.
     *
     * @param url the target URI for the inline card
     * @return a new inline card for the given URL
     * @throws IllegalArgumentException if the URL cannot be parsed
     */
    public static InlineCard inlineCard(String url) {
        return inlineCard().url(url);
    }

    /**
     * Creates an inline card for the linked content with the given URI.
     *
     * @param url the target URI for the inline card
     * @return a new inline card for the given URL
     */
    public static InlineCard inlineCard(URL url) {
        return inlineCard().url(url);
    }

    /**
     * Creates an inline card for the linked content with the given URI.
     *
     * @param url the target URI for the inline card
     * @return a new inline card for the given URL
     */
    public static InlineCard inlineCard(URI url) {
        return inlineCard().url(url);
    }

    /**
     * Creates an inline card for the linked content with the given JSON-LD description.
     *
     * @param data the link description data in <a href="https://json-ld.org/">JSON-LD</a> format
     * @return a new inline card for the given JSON-LD
     */
    public static InlineCard inlineCard(Map<String, ?> data) {
        return inlineCard().data(data);
    }

    @Override
    public InlineCard copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.INLINE_CARD;
    }

    private static InlineCard parse(Map<String, ?> map) {
        checkType(map, Type.INLINE_CARD);
        return new InlineCard(parseUrlOrData(map));
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String text = urlOrData().fold(
                url -> url,
                data -> {
                    Object url = data.get("url");
                    return (url != null) ? url.toString() : plainTextFallback(this);
                }
        );
        sb.append(text);
    }
}
