package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.node.HeadingException;
import com.atlassian.adf.model.mark.Alignment;
import com.atlassian.adf.model.mark.Indentation;
import com.atlassian.adf.model.mark.type.HeadingMark;
import com.atlassian.adf.model.mark.type.PositionMark;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.model.node.type.LayoutColumnContent;
import com.atlassian.adf.model.node.type.NestedExpandContent;
import com.atlassian.adf.model.node.type.NonNestableBlockContent;
import com.atlassian.adf.model.node.type.PanelContent;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrIntOrThrow;

/**
 * Represents a topic heading, as for the {@code <h1>} through {@code <h6>} tags used in HTML.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #h4(String) h4}("Heading 4")
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "heading",
 *     "attrs": {
 *       "level": 4
 *     },
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Heading 4"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <h4>Heading 4</h4>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/heading/">Node - heading</a>
 */
public class Heading
        extends AbstractMarkedContentNode<Heading, InlineContent, HeadingMark>
        implements DocContent, LayoutColumnContent, NestedExpandContent, NonNestableBlockContent,
        PanelContent, TableCellContent {

    static Factory<Heading> FACTORY = new Factory<>(Type.HEADING, Heading.class, Heading::parse);

    private int level;

    private Heading(int level) {
        this.level = validateLevel(level);
    }

    /**
     * @return a new, empty level 1 heading
     */
    public static Heading h1() {
        return new Heading(1);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(String content) {
        return h1().content(content);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(String... content) {
        return h1().content(content);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(InlineContent content) {
        return h1().content(content);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(InlineContent... content) {
        return h1().content(content);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(Iterable<? extends InlineContent> content) {
        return h1().content(content);
    }

    /**
     * @return a new level 1 heading with the given content
     */
    public static Heading h1(Stream<? extends InlineContent> content) {
        return h1().content(content);
    }

    /**
     * @return a new, empty level 2 heading
     */
    public static Heading h2() {
        return new Heading(2);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(String content) {
        return h2().content(content);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(String... content) {
        return h2().content(content);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(InlineContent content) {
        return h2().content(content);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(InlineContent... content) {
        return h2().content(content);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(Iterable<? extends InlineContent> content) {
        return h2().content(content);
    }

    /**
     * @return a new level 2 heading with the given content
     */
    public static Heading h2(Stream<? extends InlineContent> content) {
        return h2().content(content);
    }

    /**
     * @return a new, empty level 3 heading
     */
    public static Heading h3() {
        return new Heading(3);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(String content) {
        return h3().content(content);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(String... content) {
        return h3().content(content);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(InlineContent content) {
        return h3().content(content);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(InlineContent... content) {
        return h3().content(content);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(Iterable<? extends InlineContent> content) {
        return h3().content(content);
    }

    /**
     * @return a new level 3 heading with the given content
     */
    public static Heading h3(Stream<? extends InlineContent> content) {
        return h3().content(content);
    }

    /**
     * @return a new, empty level 4 heading
     */
    public static Heading h4() {
        return new Heading(4);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(String content) {
        return h4().content(content);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(String... content) {
        return h4().content(content);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(InlineContent content) {
        return h4().content(content);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(InlineContent... content) {
        return h4().content(content);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(Iterable<? extends InlineContent> content) {
        return h4().content(content);
    }

    /**
     * @return a new level 4 heading with the given content
     */
    public static Heading h4(Stream<? extends InlineContent> content) {
        return h4().content(content);
    }

    /**
     * @return a new, empty level 5 heading
     */
    public static Heading h5() {
        return new Heading(5);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(String content) {
        return h5().content(content);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(String... content) {
        return h5().content(content);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(InlineContent content) {
        return h5().content(content);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(InlineContent... content) {
        return h5().content(content);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(Iterable<? extends InlineContent> content) {
        return h5().content(content);
    }

    /**
     * @return a new level 5 heading with the given content
     */
    public static Heading h5(Stream<? extends InlineContent> content) {
        return h5().content(content);
    }

    /**
     * @return a new, empty level 6 heading
     */
    public static Heading h6() {
        return new Heading(6);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(String content) {
        return h6().content(content);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(String... content) {
        return h6().content(content);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(InlineContent content) {
        return h6().content(content);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(InlineContent... content) {
        return h6().content(content);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(Iterable<? extends InlineContent> content) {
        return h6().content(content);
    }

    /**
     * @return a new level 6 heading with the given content
     */
    public static Heading h6(Stream<? extends InlineContent> content) {
        return h6().content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new, empty heading with the given level
     */
    public static Heading heading(int level) {
        return new Heading(level);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, String content) {
        return heading(level).content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, String... content) {
        return heading(level).content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, InlineContent content) {
        return heading(level).content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, InlineContent... content) {
        return heading(level).content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, Iterable<? extends InlineContent> content) {
        return heading(level).content(content);
    }

    /**
     * @param level must be in the range {@code 1-6}
     * @return a new heading with the given level and content
     */
    public static Heading heading(int level, Stream<? extends InlineContent> content) {
        return heading(level).content(content);
    }

    public Heading level(int level) {
        this.level = validateLevel(level);
        return this;
    }

    /**
     * Adds the given string to this heading after first wrapping it as a {@link Text} node.
     *
     * @param content the content to add
     * @return {@code this}
     */
    public Heading content(String content) {
        return content(Text.text(content));
    }

    /**
     * Adds the given strings to this heading after first wrapping them as {@link Text} nodes.
     *
     * @param content the content to add
     * @return {@code this}
     */
    public Heading content(String... content) {
        return content(Text.text(content));
    }

    public Heading center() {
        mark(Alignment.center());
        return this;
    }

    public Heading end() {
        mark(Alignment.end());
        return this;
    }

    public Heading indentation(int level) {
        mark(Indentation.indentation(level));
        return this;
    }

    /**
     * Returns the heading level for this heading note, such as {@code 3} for an {@code <h3>} heading.
     *
     * @return the heading level for this heading note, such as {@code 3} for an {@code <h3>} heading.
     */
    public int level() {
        return level;
    }

    @Override
    public Heading mark(HeadingMark mark) {
        super.mark(mark);
        if (mark instanceof PositionMark) restrictPositionMarks((PositionMark) mark);
        return this;
    }

    @Override
    public Class<HeadingMark> markClass() {
        return HeadingMark.class;
    }

    @Override
    public Heading copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.HEADING;
    }

    @Override
    protected int markedContentNodeHashCode() {
        return level;
    }

    @Override
    protected boolean markedContentNodeEquals(Heading other) {
        return level == other.level;
    }

    @Override
    protected void appendMarkedContentNodeFields(ToStringHelper buf) {
        buf.appendField("level", level);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .let(this::addContentIfPresent)
                .let(marks::addToMap)
                .add(Key.ATTRS, map(Attr.LEVEL, level));
    }

    private static Heading parse(Map<String, ?> map) {
        checkType(map, Type.HEADING);
        int level = getAttrIntOrThrow(map, Attr.LEVEL);
        return heading(level)
                .parseOptionalContent(map, InlineContent.class)
                .parseMarks(map);
    }

    // Heading should probably override toPlainText to use appendPlainTextContent, but it doesn't because
    // the TypeScript code doesn't either. I guess they just aren't worried about headings containing
    // mention nodes?

    void disableMarks(ContentNode<?, ?> parent) {
        marks.disable(parent.elementType());
    }

    private void restrictPositionMarks(PositionMark markThatWasThereFirst) {
        Class<? extends PositionMark> allowedClass = markThatWasThereFirst.getClass();
        String reason = "Only one PositionMark is permitted, and '" + markThatWasThereFirst.elementType() +
                "' is already present";
        marks.addRule(
                mark -> allowedClass.isInstance(mark) || !(mark instanceof PositionMark),
                reason
        );
    }

    private static int validateLevel(int level) {
        if (level < 1 || level > 6) {
            throw new HeadingException.InvalidLevel(level);
        }
        return level;
    }
}
