package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.node.HardBreakException;
import com.atlassian.adf.model.node.type.CaptionContent;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;
import java.util.Optional;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttr;

/**
 * An inline node that inserts a forced line-break within a sequence of text strings.
 * It is equivalent to a {@code <br/>} in HTML.
 * <p>
 * Although it has no functional impact, {@code hardBreak} is permitted to have a text attribute.
 * If it is provided, then its value of must be {@code "\n"}.
 * </p>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Doc#doc(DocContent) doc}(
 *         {@link Paragraph#p(InlineContent[]) p}(
 *                 {@link Text#text(String) text}("Hello"),
 *                 {@link #br() br}(),
 *                 {@link Text#text(String) text}("world")
 *         )
 * )
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "version": 1,
 *     "type": "doc",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello"
 *           },
 *           {
 *             "type": "hardBreak"
 *           },
 *           {
 *             "type": "text",
 *             "text": "world"
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello<br>world</p>
 * </div>
 *
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/hardBreak/">Node - hardBreak</a>
 */
@Immutable
public class HardBreak
        extends AbstractNode<HardBreak>
        implements CaptionContent, InlineContent {

    static Factory<HardBreak> FACTORY = new Factory<>(Type.HARD_BREAK, HardBreak.class, HardBreak::parse);

    private static final String NEWLINE = "\n";
    private static final Optional<String> OPTIONAL_NEWLINE = Optional.of(NEWLINE);
    private static final HardBreak WITH_TEXT = new HardBreak(true);
    private static final HardBreak WITHOUT_TEXT = new HardBreak(false);

    private final boolean hasText;

    /**
     * @return a hard break node with the {@code "\n"} value specified as its text attribute
     */
    public static HardBreak hardBreakWithText() {
        return WITH_TEXT;
    }

    /**
     * @return a hard break node
     */
    public static HardBreak hardBreak() {
        return WITHOUT_TEXT;
    }

    /**
     * @return a hard break node
     */
    public static HardBreak br() {
        // WARNING: ForceLineBreakParser currently relies on this method returning an immutable singleton.
        return WITHOUT_TEXT;
    }

    /**
     * Returns whether the {@code "\n"} text is included in this hard break node.
     *
     * @return {@code true} if the {@code "\n"} text is included; {@code false} if it is not.
     */
    public boolean hasText() {
        return hasText;
    }

    public Optional<String> text() {
        return hasText
                ? OPTIONAL_NEWLINE
                : Optional.empty();
    }

    private HardBreak(boolean hasText) {
        this.hasText = hasText;
    }

    @Override
    public HardBreak copy() {
        return this;
    }

    @Override
    public String elementType() {
        return Type.HARD_BREAK;
    }

    @Override
    public void validate() {
    }

    @Override
    protected int nodeHashCode() {
        return Boolean.hashCode(hasText);
    }

    @Override
    protected boolean nodeEquals(HardBreak other) {
        return hasText == other.hasText;
    }

    @Override
    protected void appendNodeFields(ToStringHelper buf) {
        if (hasText) buf.appendField("hasText", true);
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .addIf(hasText, Key.ATTRS, () -> map(Attr.TEXT, NEWLINE));
    }

    private static HardBreak parse(Map<String, ?> map) {
        checkType(map, Type.HARD_BREAK);
        String text = getAttr(map, Type.TEXT, String.class).orElse(null);
        if (text == null) {
            return br();
        }
        if (NEWLINE.equals(text)) {
            return hardBreakWithText();
        }
        throw new HardBreakException.TextMustBeNewline();
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append('\n');
    }
}
