package com.atlassian.adf.model.node;

import com.atlassian.adf.model.node.AbstractNode.ToStringHelper;
import com.atlassian.adf.util.EnumParser;
import com.atlassian.adf.util.FieldMap;

import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import static com.atlassian.adf.model.Element.nonEmpty;
import static com.atlassian.adf.model.Element.nullOrNonEmpty;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrNonEmpty;
import static java.util.Objects.requireNonNull;

/**
 * A helper class that provides consistent management of extension settings for the various node types
 * that represent extensions.
 */
public class ExtensionSettings {
    private String extensionKey;
    private String extensionType;

    @Nullable
    private Map<String, ?> parameters;
    @Nullable
    private String localId;
    @Nullable
    private String text;

    private ExtensionSettings(String extensionKey, String extensionType) {
        this.extensionKey = validateExtensionKey(extensionKey);
        this.extensionType = validateExtensionType(extensionType);
    }

    public static Partial.NeedsExtensionKey<ExtensionSettings> extensionSettings() {
        return extensionSettings(Function.identity());
    }

    public static <T> Partial.NeedsExtensionKey<T> extensionSettings(Function<ExtensionSettings, T> constructor) {
        return new Partial.NeedsExtensionKey<>(constructor);
    }

    public static ExtensionSettings extensionSettings(String extensionKey, String extensionType) {
        return new ExtensionSettings(extensionKey, extensionType);
    }

    public ExtensionSettings copy() {
        ExtensionSettings copy = new ExtensionSettings(extensionKey, extensionType);
        copy.localId = localId;
        copy.text = text;
        copy.parameters = parameters;  // This map is always an immutable copy, so we don't need to deep copy it
        return copy;
    }

    public <T> T map(Function<ExtensionSettings, T> mapper) {
        requireNonNull(mapper, "mapper");
        return mapper.apply(this);
    }

    public ExtensionSettings let(Consumer<? super ExtensionSettings> effect) {
        effect.accept(this);
        return this;
    }

    public void extensionKey(String extensionKey) {
        this.extensionKey = validateExtensionKey(extensionKey);
    }

    public void extensionType(String extensionType) {
        this.extensionType = validateExtensionType(extensionType);
    }

    public void localId(@Nullable String localId) {
        this.localId = nullOrNonEmpty(localId, "localId");
    }

    public void text(@Nullable String text) {
        this.text = text;
    }

    public void parameters(@Nullable Map<String, ?> parameters) {
        this.parameters = (parameters != null)
                ? FieldMap.immutableCopy(parameters)
                : null;
    }

    public String extensionKey() {
        return extensionKey;
    }

    public String extensionType() {
        return extensionType;
    }

    public Optional<String> localId() {
        return Optional.ofNullable(localId);
    }

    public Optional<String> text() {
        return Optional.ofNullable(text);
    }

    public Optional<Map<String, ?>> parameters() {
        return Optional.ofNullable(parameters);
    }

    FieldMap toExtensionAttrs() {
        return FieldMap.map()
                .add(Node.Attr.EXTENSION_KEY, extensionKey)
                .add(Node.Attr.EXTENSION_TYPE, extensionType)
                .addIfPresent(Node.Attr.LOCAL_ID, localId)
                .addIfPresent(Node.Attr.TEXT, text)
                .addIfPresent(Node.Attr.PARAMETERS, parameters);
    }

    static ExtensionSettings parse(Map<String, ?> map) {
        String extensionKey = getAttrNonEmpty(map, Node.Attr.EXTENSION_KEY);
        String extensionType = getAttrNonEmpty(map, Node.Attr.EXTENSION_TYPE);
        ExtensionSettings data = new ExtensionSettings(extensionKey, extensionType);
        getAttr(map, Node.Attr.LOCAL_ID, String.class).ifPresent(data::localId);
        getAttr(map, Node.Attr.TEXT, String.class).ifPresent(data::text);
        getAttr(map, Node.Attr.PARAMETERS, Map.class).ifPresent(parameters -> data.parameters(unsafeCast(parameters)));
        return data;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ExtensionSettings that = (ExtensionSettings) o;
        return extensionKey.equals(that.extensionKey)
                && extensionType.equals(that.extensionType)
                && Objects.equals(parameters, that.parameters)
                && Objects.equals(localId, that.localId)
                && Objects.equals(text, that.text);
    }

    @Override
    public int hashCode() {
        return Objects.hash(extensionKey, extensionType, parameters, localId, text);
    }

    void appendExtensionFields(ToStringHelper buf) {
        buf.appendTextField(text);
        buf.appendField("extensionKey", extensionKey);
        buf.appendField("extensionType", extensionType);
        buf.appendField("localId", localId);
        buf.appendField("parameters", parameters);
    }

    private static String validateExtensionKey(String extensionKey) {
        return nonEmpty(extensionKey, "extensionKey");
    }

    private static String validateExtensionType(String extensionType) {
        return nonEmpty(extensionType, "extensionType");
    }


    /**
     * Types that represent a partially constructed extension node of some kind.
     */
    public interface Partial {
        /**
         * This partially constructed extension still needs its {@code extensionKey} attribute set.
         *
         * @param <T> the Java type of the partially constructed extension
         */
        class NeedsExtensionKey<T> {
            private final Function<ExtensionSettings, T> constructor;

            NeedsExtensionKey(Function<ExtensionSettings, T> constructor) {
                this.constructor = requireNonNull(constructor, "constructor");
            }

            public NeedsExtensionType<T> extensionKey(String extensionKey) {
                return new NeedsExtensionType<>(constructor, extensionKey);
            }
        }

        /**
         * This partially constructed extension still needs its {@code extensionType} attribute set.
         *
         * @param <T> the Java type of the partially constructed extension
         */
        class NeedsExtensionType<T> {
            private final Function<ExtensionSettings, T> constructor;
            private final String extensionKey;

            NeedsExtensionType(Function<ExtensionSettings, T> constructor, String extensionKey) {
                this.constructor = requireNonNull(constructor, "constructor");
                this.extensionKey = nonEmpty(extensionKey, "extensionKey");
            }

            public T extensionType(String extensionType) {
                ExtensionSettings settings = new ExtensionSettings(extensionKey, extensionType);
                return constructor.apply(settings);
            }
        }
    }

    public enum Layout {
        WIDE("wide"),
        FULL_WIDTH("full-width"),
        DEFAULT("default");

        static final EnumParser<Layout> PARSER = new EnumParser<>(Layout.class, Layout::layout);

        private final String layout;

        Layout(String layout) {
            this.layout = layout;
        }

        public String layout() {
            return layout;
        }
    }
}
