package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.mark.type.ExtensionMark;
import com.atlassian.adf.model.node.ExtensionSettings.Layout;
import com.atlassian.adf.model.node.type.*;
import com.atlassian.adf.util.Factory;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.atlassian.adf.model.node.ExtensionSettings.extensionSettings;
import static com.atlassian.adf.model.node.unsupported.UnsupportedNode.plainTextFallback;
import static com.atlassian.adf.util.ParserSupport.getAttr;

/**
 * Extensions provide hook points for ecosystem add-ons to integrate with how ADF content is rendered.
 * Like {@link BodiedExtension bodiedExtension} nodes, the generic {@code extension} node type is used in
 * contexts that only permit "block" content, such as the top-level {@code doc} node itself. However, there
 * are some important differences:
 * <ul>
 *     <li>{@code extension} nodes are leaves, but {@code bodiedExtension} nodes wrap other content</li>
 *     <li>{@code extension} nodes can be used in places that {@code bodiedExtension} can not, such as
 *         inside {@code tableCell}, {@code tableHeader}, or {@code bodiedExtension} itself.</li>
 * </ul>
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Extension
        extends AbstractMarkedNode<Extension, ExtensionMark>
        implements ExtensionNode<Extension, ExtensionMark>,
        DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent {

    static final Factory<Extension> FACTORY = new Factory<>(Type.EXTENSION, Extension.class, Extension::parse);

    private final ExtensionSettings settings;

    @Nullable
    private Layout layout;

    private Extension(ExtensionSettings settings) {
        this.settings = settings;
    }

    @CheckReturnValue
    public static ExtensionSettings.Partial.NeedsExtensionKey<Extension> extension() {
        return new ExtensionSettings.Partial.NeedsExtensionKey<>(Extension::new);
    }

    public static Extension extension(String extensionKey, String extensionType) {
        return extensionSettings(Extension::new)
                .extensionKey(extensionKey)
                .extensionType(extensionType);
    }

    @Override
    public Extension copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.EXTENSION;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, settings.toExtensionAttrs()
                        .addMappedIfPresent(Attr.LAYOUT, layout, Layout::layout)
                )
                .let(marks::addToMap);
    }

    @Override
    public Class<ExtensionMark> markClass() {
        return ExtensionMark.class;
    }

    @Override
    protected int markedNodeHashCode() {
        return Objects.hash(settings, layout);
    }

    @Override
    protected boolean markedNodeEquals(Extension other) {
        return layout == other.layout && settings.equals(other.settings);
    }

    @Override
    protected void appendMarkedNodeFields(ToStringHelper buf) {
        settings.appendExtensionFields(buf);
        buf.appendField("layout", layout);
    }

    @Override
    public String extensionKey() {
        return settings.extensionKey();
    }

    @Override
    public Extension extensionKey(String extensionKey) {
        settings.extensionKey(extensionKey);
        return this;
    }

    @Override
    public String extensionType() {
        return settings.extensionType();
    }

    @Override
    public Extension extensionType(String extensionType) {
        settings.extensionType(extensionType);
        return this;
    }

    @Override
    public Extension localId(@Nullable String localId) {
        settings.localId(localId);
        return this;
    }

    @Override
    public Optional<String> localId() {
        return settings.localId();
    }

    @Override
    public Extension text(@Nullable String text) {
        settings.text(text);
        return this;
    }

    @Override
    public Optional<String> text() {
        return settings.text();
    }

    @Override
    public Extension parameters(@Nullable Map<String, ?> parameters) {
        settings.parameters(parameters);
        return this;
    }

    @Override
    public Optional<Map<String, ?>> parameters() {
        return settings.parameters();
    }

    public Extension layout(@Nullable String layout) {
        return layout(Layout.PARSER.parseAllowNull(layout));
    }

    public Extension layout(@Nullable Layout layout) {
        this.layout = layout;
        return this;
    }

    public Optional<Layout> layout() {
        return Optional.ofNullable(layout);
    }

    private static Extension parse(Map<String, ?> map) {
        ExtensionSettings settings = ExtensionSettings.parse(map);
        Extension extension = new Extension(settings);
        getAttr(map, Attr.LAYOUT, String.class).ifPresent(layout -> extension.layout(Layout.PARSER.parse(layout)));
        return extension.parseMarks(map);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        String text = settings.text().orElseGet(() -> plainTextFallback(this));
        sb.append(text);
    }
}
