package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import static com.atlassian.adf.model.node.Text.text;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * The {@code decisionItem} node is a decision in a {@link DecisionList decisionList}.
 * <p>
 * Note that although this is not enforced by either the JSON schema or this library, the
 * {@code state} is expected to be either {@code "DECIDED"} or {@code "UNDECIDED"}, and an
 * {@link State enumeration} and helper functions are provided to make it easier to comply
 * with this expectation.
 *
 * <h2>Example</h2>
 * See {@link DecisionList decisionList} documentation
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class DecisionItem
        extends AbstractContentNode<DecisionItem, InlineContent> {

    static final Factory<DecisionItem> FACTORY = new Factory<>(
            Type.DECISION_ITEM,
            DecisionItem.class,
            DecisionItem::parse
    );

    private String localId;
    private String state;

    private DecisionItem(String localId, String state) {
        this.localId = requireNonNull(localId, "localId");
        this.state = requireNonNull(state, "state");
    }

    public static Partial.NeedsLocalId decisionItem() {
        return new Partial.NeedsLocalId();
    }

    @Override
    public DecisionItem copy() {
        return parse(toMap());
    }

    public DecisionItem content(String content) {
        return content(text(content));
    }

    public DecisionItem content(String... content) {
        return content(text(content));
    }

    public DecisionItem localId(String localId) {
        this.localId = requireNonNull(localId, "localId");
        return this;
    }

    public DecisionItem localId(UUID localId) {
        return localId(requireNonNull(localId, "localId").toString());
    }

    public String localId() {
        return localId;
    }

    public DecisionItem state(String state) {
        this.state = requireNonNull(state, "state");
        return this;
    }

    public DecisionItem state(State state) {
        this.state = requireNonNull(state, "state").name();
        return this;
    }

    public DecisionItem decided() {
        this.state = State.DECIDED.name();
        return this;
    }

    public DecisionItem undecided() {
        this.state = State.UNDECIDED.name();
        return this;
    }

    public String state() {
        return state;
    }

    @Override
    public String elementType() {
        return Type.DECISION_ITEM;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .let(this::addContentIfPresent)
                .add(Key.ATTRS, map()
                        .add(Attr.LOCAL_ID, localId)
                        .add(Attr.STATE, state)
                );
    }

    @Override
    protected boolean contentNodeEquals(DecisionItem other) {
        return localId.equals(other.localId)
                && state.equals(other.state);
    }

    @Override
    protected int contentNodeHashCode() {
        return Objects.hash(localId, state);
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("localId", localId);
        buf.appendField("state", state);
    }

    public static DecisionItem parse(Map<String, ?> map) {
        return decisionItem()
                .localId(getAttrOrThrow(map, Attr.LOCAL_ID, String.class))
                .state(getAttrOrThrow(map, Attr.STATE, String.class))
                .parseOptionalContent(map, InlineContent.class);
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        sb.append("<> ");
        appendPlainTextInlineContent(sb);
    }

    /**
     * Types that represent a partially constructed {@link DecisionItem decisionItem}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code decistionItem} still needs its {@code localId} attribute set.
         */
        class NeedsLocalId {
            NeedsLocalId() {
            }

            public NeedsState localId(String localId) {
                return new NeedsState(localId);
            }

            public NeedsState localId(UUID localId) {
                return new NeedsState(localId.toString());
            }
        }

        /**
         * This partially constructed {@code decisionItem} still needs its {@code state} attribute set.
         */
        class NeedsState {
            private final String localId;

            NeedsState(String localId) {
                this.localId = requireNonNull(localId, "localId");
            }

            public DecisionItem state(String state) {
                return new DecisionItem(localId, state);
            }

            public DecisionItem state(State state) {
                return new DecisionItem(localId, state.name());
            }

            public DecisionItem decided() {
                return new DecisionItem(localId, State.DECIDED.name());
            }

            public DecisionItem undecided() {
                return new DecisionItem(localId, State.UNDECIDED.name());
            }
        }
    }

    public enum State {
        DECIDED,
        UNDECIDED
    }
}
